package com.atlassian.seraph.util;

import com.atlassian.seraph.config.SecurityConfig;
import com.atlassian.seraph.config.SecurityConfigFactory;
import com.atlassian.seraph.filter.SecurityFilter;
import com.atlassian.seraph.RequestParameterConstants;

import javax.servlet.http.HttpServletRequest;
import java.net.URLEncoder;

/**
 * Utilities for login link redirection.
 */
public class RedirectUtils
{
    /**
     * Returns a login URL that would log the user in to access resource indicated by <code>request</code>.
     * <p/>
     * For instance, if <code>request</code> is for protected path "/browse/JRA-123" and the user must login before
     * accessing this resource, this method might return "/login.jsp?os_destination=%2Fbrowse%2FJRA-123". Presumably the
     * login.jsp page will redirect back to 'os_destination' once logged in.
     * <p/>
     * The returned path is derived from the <code>login.url</code> parameter in seraph-config.xml, which in the example above would be
     * "/login.jsp?os_destination={originalurl}". The '${originalurl}' token is replaced at runtime with a relative
     * or absolute path to the original resource requested by <code>request</code> ('/browse/JRA-123').
     * <p/>
     * Both the returned URL and the ${originalurl} replacement URL may be absolute or root-relative, depending on whether
     * the seraph-config.xml <code>login.url</code> parameter is.  This allows for redirection to external <acronym title="Single Sign-on">SSO</acronym>
     * apps, which are passed an absolute path to the originally requested resource.
     * <p/>
     * No actual permission checks are performed to determine whether the user needs to log in to access the resource. The
     * caller is assumed to have done this before calling this method.
     *
     * @param request The original request made by the user for a resource.
     * @return A root-relative or absolute URL of a login link that would log the user in to access the resource.
     */
    public static String getLoginUrl(HttpServletRequest request)
    {
        SecurityConfig securityConfig = SecurityConfigFactory.getInstance();
        String loginURL = securityConfig.getLoginURL();
        return getLoginURL(loginURL, request);
    }

    /**
     * Returns a login URL that would log the user in to access resource indicated by <code>request</code>.
     * Identical to {@link #getLoginUrl(javax.servlet.http.HttpServletRequest)}, except uses the 'link.login.url'
     * parameter in seraph-config.xml instead of 'login.url', which allows for different login pages depending on whether
     * invoked from a link ("link.login.url") or from a servlet filter that intercepted a request ("login.url").
     *
     * @see #getLoginUrl(javax.servlet.http.HttpServletRequest) for parameters, etc
     */
    public static String getLinkLoginURL(HttpServletRequest request)
    {
        SecurityConfig securityConfig = SecurityConfigFactory.getInstance();
        String loginURL = securityConfig.getLinkLoginURL();
        return getLoginURL(loginURL, request);
    }


    private static String getLoginURL(String loginURL, HttpServletRequest request)
    {
        boolean externalLoginLink = isExternalLoginLink(loginURL);
        loginURL = replaceOriginalURL(loginURL, request, externalLoginLink);
        if (externalLoginLink)
        {
            return loginURL;
        }
        else
        {
            return request.getContextPath() + loginURL;
        }
    }

    private static boolean isExternalLoginLink(String loginURL)
    {
        return (loginURL.indexOf("://") != -1);
    }

    /**
     * Replace ${originalurl} token in a string with a URL for a Request.
     */
    private static String replaceOriginalURL(final String loginURL, final HttpServletRequest request, boolean external)
    {
        final int i = loginURL.indexOf("${originalurl}");
        if (i != -1)
        {
            final String originalURL = getOriginalURL(request, external);
            String osDest = request.getParameter(RequestParameterConstants.OS_DESTINATION);
            return loginURL.substring(0, i) + ((osDest != null) ? URLEncoder.encode(osDest) : URLEncoder.encode(originalURL) )+ loginURL.substring(i + "${originalurl}".length());
        }
        else
            return loginURL;
    }

    /**
     * Recreate a URL from a Request.
     */
    private static String getOriginalURL(HttpServletRequest request, boolean external)
    {
        String originalURL = (String) request.getAttribute(SecurityFilter.ORIGINAL_URL);
        if (originalURL != null)
        {
            if (external)
                return getServerNameAndPath(request) + originalURL;
            else
                return originalURL;
        }

        if (external)
            return request.getRequestURL() + (request.getQueryString() == null ? "" : "?" + request.getQueryString());
        else
            return request.getServletPath() +
                (request.getPathInfo() == null ? "" : request.getPathInfo()) +
                (request.getQueryString() == null ? "" : "?" + request.getQueryString());

    }

    /**
     * Reconstruct the request URL from a HttpServletRequest.
     */
    public static String getServerNameAndPath(HttpServletRequest request)
    {
        StringBuffer buf = new StringBuffer();
        buf.append(request.getScheme()).
                append("://").
                append(request.getServerName());
        if (! (("http".equals(request.getScheme()) && request.getServerPort() == 80) || ("https".equals(request.getScheme()) && request.getServerPort() == 443)))
        {
            buf.append(":").append(request.getServerPort());
        }
        buf.append(request.getContextPath());
        return buf.toString();
    }

    /**
     * Checks whether the request contains a basicAuth parameter in its queryString
     */
    public static boolean isBasicAuthentication(HttpServletRequest request, String basicAuth)
    {
        String queryString = request.getQueryString();
        if (queryString == null || queryString.equals(""))
            return false;

        String authString = basicAuth + "=" + SecurityConfig.BASIC_AUTH;
        if (queryString.indexOf(authString) != -1)
            return true;
        else
            return false;
    }

    /**
     * Appends the path to the context, dealing with any missing slashes properly.
     * Does NOT resolve the path using URL resolution rules. Does NOT attempt to
     * normalise the resulting URL.
     *
     * @param context a context path as returned by HttpServletRequest.getContextPath, e.g. "/confluence".
     * If null, it will be treated as the default context ("").
     * @param path a path to be appended to the context, e.g. "/homepage.action".
     * If this is null, the context will be returned if it is non-null, otherwise the empty string will be returned.
     * @return a URL of the given path within the context, e.g. "/confluence/homepage.action".
     */
    public static String appendPathToContext(String context, String path)
    {
        if (context == null) context = "";
        if (path == null) return context;

        StringBuffer result = new StringBuffer(context);
        if (!context.endsWith("/"))
            result.append("/");

        String pathToAppend = path;
        if (pathToAppend.startsWith("/"))
            pathToAppend = pathToAppend.substring(1);

        result.append(pathToAppend);
        return result.toString();
    }
}
