package com.atlassian.seraph.service;

import com.atlassian.seraph.util.PathMapper;
import com.atlassian.seraph.util.CachedPathMapper;
import com.atlassian.seraph.SecurityService;
import com.atlassian.seraph.config.SecurityConfig;
import com.opensymphony.util.ClassLoaderUtil;
import org.apache.log4j.Category;
import org.apache.commons.collections.map.LRUMap;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

import javax.servlet.http.HttpServletRequest;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import java.net.URL;
import java.util.*;
import java.io.IOException;

/**
 * Configures Seraph based on WebWork 1.x configuration file actions.xml.
 * <p/>
 * Takes two optional init parameters:
 * <dl>
 * <dt>action.extension (default: "action")</dt>
 * <dd>The extension of action URLs which are intercepted by this service.
 * If this parameter is not specified, the default is the WebWork default 'action'.
 * (This should match your servlet mapping in web.xml.)</dd>
 * <dt>actions.xml.file (default: "actions")</dt>
 * <dd>The location on the classpath of your actions.xml file (WebWork 1.x style).
 * If this parameter is not specified, the default is the WebWork 1.x default 'actions'.
 * (This should match the value configured for 'webwork.configuration.xml' in
 * webwork.properties.)</dd>
 * </dl>
 * In actions.xml (or other actions XML file, as specified by the init-param) specify
 * roles required per action or command:
 * <pre>
 *   &lt;action name="project.ViewProject" alias="ViewProject" roles-required="RoleFoo, RoleBar"&gt;
 * or
 *   &lt;command name="Delete" alias="DeleteProject" roles-required="RoleBat"&gt;
 * </pre>
 * Roles can be separated by commas and spaces.
 */
public class WebworkService implements SecurityService
{
    private static final Category log = Category.getInstance(WebworkService.class);
    private static final String ROLES_REQUIRED_ATTR = "roles-required";
    private static final String ACTION_EXTENSION_INIT_PARAM = "action.extension";
    private static final String ACTIONS_XML_FILE_INIT_PARAM = "actions.xml.file";

    // Same as the WebWork default (i.e. action URLs end with '.action')
    private static final String ACTION_EXTENSION_DEFAULT = "action";

    // Same as the WebWork 1.x default (i.e. actions.xml is the mapping file)
    private static final String ACTIONS_XML_FILE_DEFAULT = "actions";

    // used to check which actions match the current path
    // This class only uses the "get:" method of the pathmapper, so initialise the map
    // that caches its results to a large value. The getAll method of the PathMapper is not used
    // by this class so make the second caching map small.
    private PathMapper actionMapper = new CachedPathMapper(new LRUMap(500), new LRUMap(10));

    // maps current action to roles required
    private Map rolesMap = new HashMap();

    public void init(Map params, SecurityConfig config)
    {
        try
        {
            // the extension of webwork actions (should match web.xml servlet mapping)
            String extension = (String) params.get(ACTION_EXTENSION_INIT_PARAM);
            if (extension == null) extension = ACTION_EXTENSION_DEFAULT;

            // the XML file containing the action mappings (should match 'webwork.configuration.xml' property)

            String actionsXmlFile = (String) params.get(ACTIONS_XML_FILE_INIT_PARAM);
            if (actionsXmlFile == null) actionsXmlFile = ACTIONS_XML_FILE_DEFAULT;

            configureActionMapper(extension, actionsXmlFile);
        }
        catch (RuntimeException e)
        {
            log.error("Failed to initialise WebworkService", e);
        }
    }

    private void configureActionMapper(String extension, String actionsXmlFile)
    {
        org.w3c.dom.Document doc = parseActionsXmlFile(actionsXmlFile);

        // Get list of actions
        NodeList actions = doc.getElementsByTagName("action");

        String rootRolesRequired = overrideRoles(null, doc.getDocumentElement());

        // Build list of views
        for (int i = 0; i < actions.getLength(); i++)
        {
            Element action = (Element) actions.item(i);
            String actionName = action.getAttribute("name");
            String actionAlias = action.getAttribute("alias");
            final String actionRolesRequired = overrideRoles(rootRolesRequired, action);

            if (actionRolesRequired != null)
            {

                if (actionAlias != null)
                {
                    actionMapper.put(actionAlias, "/" + actionAlias + "." + extension);
                    rolesMap.put(actionAlias, actionRolesRequired);
                    actionMapper.put(actionAlias + "!*", "/" + actionAlias + "!*." + extension);
                    rolesMap.put(actionAlias + "!*", actionRolesRequired);
                }

                if (actionName != null)
                {
                    actionMapper.put(actionName, "/" + actionName + "." + extension);
                    rolesMap.put(actionName, actionRolesRequired);
                    actionMapper.put(actionName + "!*", "/" + actionName + "!*." + extension);
                    rolesMap.put(actionName + "!*", actionRolesRequired);
                }
            }

            // Get list of commands
            NodeList commands = action.getElementsByTagName("command");
            for (int j = 0; j < commands.getLength(); j++)
            {
                Element command = (Element) commands.item(j);
                String cmdRolesRequired = overrideRoles(actionRolesRequired, command);

                String commandAlias = command.getAttribute("alias");

                if (commandAlias != null && cmdRolesRequired != null)
                {
                    actionMapper.put(commandAlias, "/" + commandAlias + "." + extension);
                    rolesMap.put(commandAlias, cmdRolesRequired);
                }
            }
        }
    }

    private org.w3c.dom.Document parseActionsXmlFile(String actionsXmlFile)
    {
        DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
        URL fileUrl = ClassLoaderUtil.getResource(actionsXmlFile + ".xml", this.getClass());

        if (fileUrl == null)
            fileUrl = ClassLoaderUtil.getResource("/" + actionsXmlFile + ".xml", this.getClass());

        if (fileUrl == null)
            throw new IllegalArgumentException("No such XML file:/" + actionsXmlFile + ".xml");

        try
        {
            return factory.newDocumentBuilder().parse(fileUrl.toString());
        }
        catch (SAXException e)
        {
            throw new RuntimeException(e);
        }
        catch (IOException e)
        {
            throw new RuntimeException(e);
        }
        catch (ParserConfigurationException e)
        {
            throw new RuntimeException(e);
        }
    }

    /**
     * Returns newRolesRequired if it isn't empty, and rolesRequired otherwise.
     */
    private String overrideRoles(String rolesRequired, Element action)
    {
        if (action.hasAttribute(ROLES_REQUIRED_ATTR))
        {
            return action.getAttribute(ROLES_REQUIRED_ATTR);
        }
        else
        {
            return rolesRequired;
        }
    }

    public void destroy()
    {
    }

    public Set getRequiredRoles(HttpServletRequest request)
    {
        Set requiredRoles = new HashSet();

        String currentURL = request.getRequestURI();

        int lastSlash = currentURL.lastIndexOf('/');
        String targetURL;

        // then check webwork mappings
        if (lastSlash > -1)
        {
            targetURL = currentURL.substring(lastSlash);
        }
        else
        {
            targetURL = currentURL;
        }

        String actionMatch = actionMapper.get(targetURL);

        if (actionMatch != null)
        {
            String rolesStr = (String) rolesMap.get(actionMatch);

            StringTokenizer st = new StringTokenizer(rolesStr, ", ");
            while (st.hasMoreTokens())
            {
                requiredRoles.add(st.nextToken());
            }
        }

        return requiredRoles;
    }
}
