package com.atlassian.seraph.filter;

import com.atlassian.security.auth.trustedapps.ApplicationCertificate;
import com.atlassian.security.auth.trustedapps.CurrentApplication;
import com.atlassian.security.auth.trustedapps.DefaultEncryptedCertificate;
import com.atlassian.security.auth.trustedapps.InvalidCertificateException;
import com.atlassian.security.auth.trustedapps.TransportErrorMessage;
import com.atlassian.security.auth.trustedapps.TrustedApplication;
import com.atlassian.security.auth.trustedapps.TrustedApplicationUtils;
import com.atlassian.security.auth.trustedapps.TrustedApplicationsManager;
import com.atlassian.security.auth.trustedapps.UserResolver;
import com.atlassian.seraph.auth.DefaultAuthenticator;
import com.atlassian.seraph.auth.RoleMapper;
import com.atlassian.seraph.config.SecurityConfigFactory;

import org.apache.log4j.Logger;
import org.bouncycastle.util.encoders.Base64;

import java.io.IOException;
import java.io.OutputStreamWriter;
import java.io.UnsupportedEncodingException;
import java.io.Writer;
import java.security.Principal;
import java.security.PublicKey;

import javax.servlet.Filter;
import javax.servlet.FilterChain;
import javax.servlet.FilterConfig;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

/**
 * This filter serves two purposes: 1. Authenticates requests from trusted applications if the right certificate is
 * present in the request 2. Returns the UID and public key of this application upon request so other servers can
 * establish trusted relationship with this application as a client
 * <p>
 * For the first purpose, the filter will intercept any calls to a page '/admin/appTrustCertificate'. Directory
 * structure of the request will be ignored. The returned page will contain 2 lines:
 * <ul>
 * <li> ID </li>
 * <li> public key BASE64 encoded </li>
 * </ul>
 * <p>
 * For the second purpose the following header parameters must be present and valid:
 * {@link CurrentApplication#HEADER_TRUSTED_APP_CERT} {@link CurrentApplication#HEADER_TRUSTED_APP_ID}
 * <p>
 * If the authentication should fail a message will be set in the response header:
 * {@link CurrentApplication#HEADER_TRUSTED_APP_ERROR}
 */
public class TrustedApplicationsFilter implements Filter
{
    private static final Logger log = Logger.getLogger(TrustedApplicationsFilter.class);

    private static final class Status
    {
        static final String ERROR = "ERROR";
        static final String OK = "OK";
    }

    private final CertificateServer certificateServer;
    private final Authenticator authenticator;

    private FilterConfig filterConfig = null;

    /*
     * used to get the static seraph-config.xml driven RoleMapper
     */
    // /CLOVER:OFF
    public TrustedApplicationsFilter(TrustedApplicationsManager appManager, UserResolver resolver)
    {
        this(appManager, resolver, SecurityConfigFactory.getInstance().getRoleMapper());
    }

    // /CLOVER:ON

    public TrustedApplicationsFilter(TrustedApplicationsManager appManager, UserResolver resolver, RoleMapper roleMapper)
    {
        this(new CertificateServerImpl(appManager), new AuthenticatorImpl(appManager, resolver, roleMapper));
    }

    TrustedApplicationsFilter(CertificateServer certificateServer, Authenticator authenticator)
    {
        notNull("certificateServer", certificateServer);
        notNull("authenticator", authenticator);
        this.certificateServer = certificateServer;
        this.authenticator = authenticator;
    }

    public void doFilter(ServletRequest req, ServletResponse res, FilterChain chain) throws IOException, ServletException
    {
        // if this is a certificate request
        // serve the certificate back and return
        HttpServletRequest request = (HttpServletRequest) req;
        HttpServletResponse response = (HttpServletResponse) res;

        if (getPathInfo(request).endsWith("/admin/appTrustCertificate"))
        {
            response.setContentType("text/plain");
            certificateServer.writeCertificate(new OutputStreamWriter(response.getOutputStream()));
            return;
        }

        boolean isTrustedAppCall = false;
        // for all other requests - let the parent class do its job
        if (request.getAttribute(BaseLoginFilter.OS_AUTHSTATUS_KEY) == null)
        {
            String status = authenticate((HttpServletRequest) req, (HttpServletResponse) res);
            if (BaseLoginFilter.LOGIN_SUCCESS.equals(status))
            {
                request.setAttribute(BaseLoginFilter.OS_AUTHSTATUS_KEY, status);
                response.setHeader(TrustedApplicationUtils.Header.Response.STATUS, Status.OK);
                isTrustedAppCall = true;
            }
        }
        try
        {
            chain.doFilter(request, res);
        }
        finally
        {
            if (isTrustedAppCall && request.getSession(false) != null)
            {
                request.getSession().invalidate();
            }
        }
    }

    public String authenticate(HttpServletRequest request, HttpServletResponse response)
    {
        final Authenticator.Result result = authenticator.authenticate(request);

        switch (result.getStatus().getOrdinal())
        {
            case Authenticator.Result.Status.Constants.NO_ATTEMPT:
                return BaseLoginFilter.LOGIN_NOATTEMPT;

            case Authenticator.Result.Status.Constants.FAILED:
                setFailureHeader(response, result.getMessage());
                return BaseLoginFilter.LOGIN_FAILED;

            case Authenticator.Result.Status.Constants.ERROR:
                setFailureHeader(response, result.getMessage());
                return BaseLoginFilter.LOGIN_ERROR;

            case Authenticator.Result.Status.Constants.SUCCESS:
                request.getSession().setAttribute(DefaultAuthenticator.LOGGED_IN_KEY, result.getUser());
                request.getSession().setAttribute(DefaultAuthenticator.LOGGED_OUT_KEY, null);
                return BaseLoginFilter.LOGIN_SUCCESS;

                // /CLOVER:OFF
            default:
                throw new IllegalStateException("Unknown result: " + result.getStatus().getOrdinal());
                // /CLOVER:ON
        }
    }

    protected String getPathInfo(HttpServletRequest request)
    {
        String context = request.getContextPath();
        String uri = request.getRequestURI();
        if (context != null && context.length() > 0)
        {
            return uri.substring(context.length());
        }
        else
        {
            return uri;
        }
    }

    private void setFailureHeader(HttpServletResponse response, String message)
    {
        response.setHeader(TrustedApplicationUtils.Header.Response.STATUS, Status.ERROR);
        response.addHeader(TrustedApplicationUtils.Header.Response.ERROR, message);
        if (log.isInfoEnabled())
        {
            log.info(message, new RuntimeException(message));
        }
    }

    public void init(FilterConfig config)
    {
        this.filterConfig = config;
    }

    public void destroy()
    {
        filterConfig = null;
    }

    /** @deprecated Not needed in latest version of Servlet 2.3 API */
    public FilterConfig getFilterConfig()
    {
        return filterConfig;
    }

    /** @deprecated Not needed in latest version of Servlet 2.3 API - replaced by init(). */
    public void setFilterConfig(FilterConfig filterConfig)
    {
        if (filterConfig != null) // it seems that Orion 1.5.2 calls this with a null config.
        {
            init(filterConfig);
        }
    }

    /**
     * serve the CurrentApplication's certificate
     */
    interface CertificateServer
    {
        void writeCertificate(Writer writer) throws IOException;
    }

    static class CertificateServerImpl implements CertificateServer
    {
        final TrustedApplicationsManager appManager;

        CertificateServerImpl(TrustedApplicationsManager appManager)
        {
            this.appManager = appManager;
        }

        public void writeCertificate(Writer writer) throws IOException
        {
            CurrentApplication currentApplication = appManager.getCurrentApplication();
            PublicKey publicKey = currentApplication.getPublicKey();

            try
            {
                writer.write(currentApplication.getID());
                writer.write("\n");

                byte[] key = publicKey.getEncoded();
                writer.write(new String(Base64.encode(key), TrustedApplicationUtils.Constant.CHARSET_NAME));
                writer.write("\n");
                writer.write(TrustedApplicationUtils.Constant.VERSION.toString());
                writer.write("\n");
                writer.write(TrustedApplicationUtils.Constant.MAGIC);
                writer.flush();
            }
            catch (UnsupportedEncodingException ex)
            {
                throw new AssertionError(ex);
            }
            catch (IOException e)
            {
                throw new RuntimeException(e);
            }
        }
    }

    /**
     * Authenticate a TrustedApplication request
     */
    interface Authenticator
    {
        Result authenticate(HttpServletRequest request);

        static class Result
        {
            private final Status status;
            private final TransportErrorMessage message;
            private final Principal user;

            Result(Status status)
            {
                this(status, null, null);
            }

            Result(Status status, TransportErrorMessage message)
            {
                this(status, message, null);

                notNull("message", message);
            }

            Result(Status status, Principal principal)
            {
                this(status, null, principal);

                notNull("principal", principal);
            }

            Result(Status status, TransportErrorMessage message, Principal user)
            {
                if (status == null)
                {
                    throw new IllegalArgumentException("status");
                }
                this.status = status;
                this.message = message;
                this.user = user;
            }

            public Status getStatus()
            {
                return status;
            }

            public String getMessage()
            {
                return message.toString();
            }

            public Principal getUser()
            {
                return user;
            }

            static final class Status
            {
                /**
                 * Necessary to declare these as int constants as javac is too dumb to realise that a final member of a
                 * final class's static constants is a constant
                 */
                static final class Constants
                {
                    static final int SUCCESS = 0;
                    static final int FAILED = 1;
                    static final int ERROR = 2;
                    static final int NO_ATTEMPT = 3;
                }

                static final Status SUCCESS = new Status(Constants.SUCCESS, "success");
                static final Status FAILED = new Status(Constants.FAILED, "failed");
                static final Status ERROR = new Status(Constants.ERROR, "error");
                static final Status NO_ATTEMPT = new Status(Constants.NO_ATTEMPT, "no attempt");

                private final int ordinal;
                private final String name;

                private Status(int ordinal, String name)
                {
                    this.ordinal = ordinal;
                    this.name = name;
                }

                int getOrdinal()
                {
                    return ordinal;
                }

                public String toString()
                {
                    return name;
                }
            }

            static class NoAttempt extends Result
            {
                NoAttempt()
                {
                    super(Status.NO_ATTEMPT);
                }
            }

            static class Error extends Result
            {
                Error(TransportErrorMessage message)
                {
                    super(Status.ERROR, message);
                }
            }

            static class Failure extends Result
            {
                Failure(TransportErrorMessage message)
                {
                    super(Status.FAILED, message);
                }
            }

            static class Success extends Result
            {
                public Success(Principal principal)
                {
                    super(Status.SUCCESS, principal);
                }
            }
        }
    }

    static class AuthenticatorImpl implements Authenticator
    {
        final TrustedApplicationsManager appManager;
        final UserResolver resolver;
        final RoleMapper roleMapper;

        AuthenticatorImpl(TrustedApplicationsManager appManager, UserResolver resolver, RoleMapper roleMapper)
        {
            this.appManager = appManager;
            this.resolver = resolver;
            this.roleMapper = roleMapper;
        }

        public Result authenticate(HttpServletRequest request)
        {
            final String certStr = request.getHeader(TrustedApplicationUtils.Header.Request.CERTIFICATE);
            if (isBlank(certStr))
            {
                return new Result.NoAttempt();
            }

            final String id = request.getHeader(TrustedApplicationUtils.Header.Request.ID);
            if (isBlank(id))
            {
                return new Result.Error(new TransportErrorMessage.ApplicationIdNotFoundInRequest());
            }

            final String key = request.getHeader(TrustedApplicationUtils.Header.Request.SECRET_KEY);
            if (isBlank(key))
            {
                return new Result.Error(new TransportErrorMessage.SecretKeyNotFoundInRequest());
            }

            final String magicNumber = request.getHeader(TrustedApplicationUtils.Header.Request.MAGIC);
            // magic number validation is only done from protocol version 2, version 1 had no version header
            final String version = request.getHeader(TrustedApplicationUtils.Header.Request.VERSION);
            final Integer protocolVersion;
            try
            {
                protocolVersion = (!isBlank(version)) ? new Integer(version) : null;
            }
            catch (NumberFormatException e)
            {
                return new Result.Error(new TransportErrorMessage.BadProtocolVersion(version));
            }

            // note, this code only handles non-null version - doesn't differentiate between 1 & 2
            if (protocolVersion != null)
            {
                if (isBlank(magicNumber))
                {
                    return new Result.Error(new TransportErrorMessage.MagicNumberNotFoundInRequest());
                }
            }

            TrustedApplication app = appManager.getTrustedApplication(id);
            if (app == null)
            {
                return new Result.Failure(new TransportErrorMessage.ApplicationUnknown(id));
            }

            final ApplicationCertificate certificate;
            try
            {
                certificate = app.decode(new DefaultEncryptedCertificate(id, key, certStr, protocolVersion, magicNumber), request);
            }
            catch (InvalidCertificateException ex)
            {
                log.warn("Failed to login trusted application: " + app.getID() + " due to: " + ex);
                // debug for stacktrace, no need for isDebugEnabled check as there is no string concatenation
                log.debug("Failed to login trusted application cause", ex);
                return new Result.Error(ex.getTransportErrorMessage());
            }
            final Principal user = resolver.resolve(certificate);
            if (user == null)
            {
                log.warn("User '" + certificate.getUserName() + "' referenced by trusted application: '" + app.getID() + "' is not found.");
                return new Result.Failure(new TransportErrorMessage.UserUnknown(certificate.getUserName()));
            }
            else if (!roleMapper.canLogin(user, request))
            {
                // user exists but is not allowed to login
                log.warn("User '" + certificate.getUserName() + "' referenced by trusted application: '" + app.getID() + "' cannot login.");
                return new Result.Failure(new TransportErrorMessage.PermissionDenied());
            }

            return new Result.Success(user);
        }
    }

    private static boolean isBlank(String input)
    {
        return (input == null) || input.trim().length() == 0;
    }

    private static void notNull(String name, Object notNull)
    {
        if (notNull == null)
        {
            throw new IllegalArgumentException(name + " should not be null");
        }
    }
}