package com.atlassian.seraph.filter;

import java.io.IOException;
import java.security.Principal;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;

import javax.servlet.Filter;
import javax.servlet.FilterChain;
import javax.servlet.FilterConfig;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.log4j.Category;

import com.atlassian.seraph.SecurityService;
import com.atlassian.seraph.auth.AuthenticationContext;
import com.atlassian.seraph.config.SecurityConfig;
import com.atlassian.seraph.config.SecurityConfigFactory;
import com.atlassian.seraph.util.RedirectUtils;

/**
 * The SecurityFilter determines which roles are required for a given request by querying all of the Services.
 *
 * @see SecurityService
 */
public class SecurityFilter implements Filter
{
    private FilterConfig config = null;
    private SecurityConfig securityConfig = null;

    private static final Category log = Category.getInstance(SecurityFilter.class);
    private static final String ALREADY_FILTERED = "os_securityfilter_already_filtered";
    public static final String ORIGINAL_URL = "atlassian.core.seraph.original.url";

    public void init(FilterConfig config)
    {
        // log.debug("SecurityFilter.init");
        this.config = config;

        String configFileLocation = null;

        if (config.getInitParameter("config.file") != null)
        {
            configFileLocation = config.getInitParameter("config.file");
            log.debug("Security config file location: " + configFileLocation);
        }

        securityConfig = SecurityConfigFactory.getInstance(configFileLocation);
        config.getServletContext().setAttribute(SecurityConfig.STORAGE_KEY, securityConfig);
    }

    public void destroy()
    {
        // log.debug("SecurityFilter.destroy");
        securityConfig.destroy();
        securityConfig = null;
        config = null;
    }

    /**
     * @deprecated Not needed in latest version of Servlet 2.3 API
     */
    // NOTE: Filter doesn't work with Orion 1.5.2 without this method
    public FilterConfig getFilterConfig()
    {
        return config;
    }

    /**
     * @deprecated Not needed in latest version of Servlet 2.3 API - replaced by init().
     */
    // NOTE: Filter doesn't work with Orion 1.5.2 without this method
    public void setFilterConfig(FilterConfig filterConfig)
    {
        if (filterConfig != null) //it seems that Orion 1.5.2 calls this with a null config.
            init(filterConfig);
    }

    public void doFilter(ServletRequest req, ServletResponse res, FilterChain chain)
            throws IOException, ServletException
    {
        if (req.getAttribute(ALREADY_FILTERED) != null || !getSecurityConfig().getController().isSecurityEnabled())
        {
            chain.doFilter(req, res);
            return;
        }
        else
        {
            req.setAttribute(ALREADY_FILTERED, Boolean.TRUE);
        }

        // Try and get around Orion's bug when redeploying
        // it seems that filters are not called in the correct order
        if (req.getAttribute(BaseLoginFilter.ALREADY_FILTERED) == null)
        {
            log.warn("LoginFilter not yet applied to this request - terminating filter chain");
            return;
        }

        HttpServletRequest request = (HttpServletRequest) req;
        HttpServletResponse response = (HttpServletResponse) res;

        String originalURL = request.getServletPath() +
                (request.getPathInfo() == null ? "" : request.getPathInfo()) +
                (request.getQueryString() == null ? "" : "?" + request.getQueryString());


        // store the original URL as a request attribute anyway - often useful for pages to access it (ie login links)
        request.setAttribute(SecurityFilter.ORIGINAL_URL, originalURL);
        if (log.isDebugEnabled())
        	log.debug("Storing the originally requested URL (" + SecurityFilter.ORIGINAL_URL + "=" + originalURL + ")");

        Set requiredRoles = new HashSet();

        // loop through loaded services and get required roles
        for (Iterator iterator = getSecurityConfig().getServices().iterator(); iterator.hasNext();)
        {
            SecurityService service = (SecurityService) iterator.next();

            Set serviceRoles = service.getRequiredRoles(request);
            requiredRoles.addAll(serviceRoles);
        }

        if (log.isDebugEnabled()) {
            log.debug("requiredRoles = " + requiredRoles);
        }

        // whether this URL needs authorisation
        boolean needAuth = false;

        // try to get the user (for cookie logins)
        Principal user = getSecurityConfig().getAuthenticator().getUser(request, response);

        // only redirect if we don't use basic HTTP authentication
        boolean basicAuthentication = RedirectUtils.isBasicAuthentication(request, getSecurityConfig().getAuthType());
        // if we are doing basic authentication, then we don't want to continue the filter chain - we need to commit
        // the response to allow for the authentication challenge mechanism
        if (basicAuthentication && user == null)
        {
        	log.debug("Basic authentication requested.");
            return;
        }

        // set the user in the context
        getAuthenticationContext().setUser(user);

        // check if the current user has all required permissions
        // if there is no current user, request.isUserInRole() always returns false so this works
        for (Iterator iterator = requiredRoles.iterator(); iterator.hasNext();)
        {
            String role = (String) iterator.next();

            // this isUserInRole method is only used here and 'd be better off replaced by getRoleMapper().hasRole(user, request, role)) since we have the user already
            // was : if (!securityConfig.getAuthenticator().isUserInRole(request, role))
            if (!getSecurityConfig().getRoleMapper().hasRole(user, request, role))
            {
                log.info("User '" + user + "' needs (and lacks) role '" + role + "' to access " + originalURL);
                needAuth = true;
            }
        }

        // check if we're at the signon page, in which case do not auth
        if (request.getServletPath() != null && request.getServletPath().equals(getSecurityConfig().getLoginURL()))
        {
        	log.debug("Login page requested so no additional authorization required.");
            needAuth = false;
        }

        // if we need to authenticate, store current URL and forward
        if (needAuth)
        {
            if (log.isDebugEnabled())
                log.debug("Need Authentication: Redirecting to: " + getSecurityConfig().getLoginURL() + " from: " + originalURL);

            request.getSession().setAttribute(getSecurityConfig().getOriginalURLKey(), originalURL);
            // only redirect if we can. if isCommited==true, there might have been a redirection requested by a LoginInterceptor, for instance.
            if (!response.isCommitted())
            {
                response.sendRedirect(RedirectUtils.getLoginUrl(request));
            }
            return;
        }
        else
        {
            try
            {
                chain.doFilter(req, res);
            }
            finally
            {
                // clear the user from the context
                getAuthenticationContext().clearUser();
            }
        }
    }

    protected SecurityConfig getSecurityConfig() {
        // ?? is this any useful, since we already initalize this in the init method ??
        if (securityConfig == null)
        {
            securityConfig = (SecurityConfig) config.getServletContext().getAttribute(SecurityConfig.STORAGE_KEY);
        }
        return securityConfig;
    }

    protected AuthenticationContext getAuthenticationContext() {
        return getSecurityConfig().getAuthenticationContext();
    }

}
