package com.atlassian.security.auth.trustedapps;

import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;
import java.util.StringTokenizer;

/**
 * simple list based implementation. Matches using exact matches or wildcards like: 192.168.*.* or 192.145.372.*
 */
public class DefaultIPMatcher implements IPMatcher
{
    private static class AddressMask
    {
        private final int address;
        private final int mask;

        public AddressMask(int address, int mask)
        {
            this.address = address;
            this.mask = mask;
        }

        public boolean matches(int otherAddress)
        {
            return address == (otherAddress & mask);
        }

        static AddressMask create(int[] pattern)
        {
            int address = 0;
            int mask = 0;

            for (int i = 0; i < pattern.length; i++)
            {
                address = address << 8;
                mask = mask << 8;

                if (pattern[i] != -1)
                {
                    address = address | pattern[i];
                    mask = mask | 0xFF;
                }
            }

            return new AddressMask(address, mask);
        }
    }

    private static final String WILDCARD = "*";

    private final List /*<AddressMask>*/ addressMasks;

    /**
     * Main ctor.
     * 
     * @param patterns the Set<String> of allowed pattern Strings
     * @throws IPAddressFormatException if the pattern does not represent a valid IP address
     */
    public DefaultIPMatcher(Set patterns) throws IPAddressFormatException
    {
        this.addressMasks = new LinkedList();
        for (Iterator iterator = patterns.iterator(); iterator.hasNext();)
        {
            String patternStr = (String) iterator.next();
            int[] pattern = parsePatternString(patternStr);
            addressMasks.add(AddressMask.create(pattern));
        }
    }

    private int[] parsePatternString(String patternStr)
    {
        int[] pattern = new int[4];
        StringTokenizer st = new StringTokenizer(patternStr, ".");
        if (st.countTokens() != 4)
        {
            throw new IPAddressFormatException(patternStr);
        }

        for (int i = 0; i < 4; i++)
        {
            final String token = st.nextToken().trim();
            if (WILDCARD.equals(token))
            {
                pattern[i] = -1;
            }
            else
            {
                try
                {
                    int value = Integer.valueOf(token).intValue();

                    if ((value < 0) || value > 255)
                    {
                        throw new IPAddressFormatException(patternStr);
                    }

                    pattern[i] = value;
                }
                catch (NumberFormatException e)
                {
                    throw new IPAddressFormatException(patternStr);
                }
            }
        }
        return pattern;
    }

    public boolean match(String ipAddress)
    {
        if (addressMasks.isEmpty())
        {
            return true;
        }

        int address = toAddress(ipAddress);

        for (Iterator iterator = addressMasks.iterator(); iterator.hasNext();)
        {
            AddressMask addressMask = (AddressMask) iterator.next();
            if (addressMask.matches(address))
            {
                return true;
            }
        }
        return false;
    }

    private int toAddress(String ipAddress)
    {
        int address = 0;
        int[] parsedIPAddr = parsePatternString(ipAddress);
        for (int i = 0; i < parsedIPAddr.length; i++)
        {
            address = address << 8;
            address = address | parsedIPAddr[i];
        }
        return address;
    }
}