package com.atlassian.security.password;

/**
 * Encodes passwords for storage in an application and allows verification of raw
 * passwords against the stored versions. The actual encoding mechanism depends
 * on the implementation.
 *
 * @see DefaultPasswordEncoder for a useful starting point for new implementations
 */
public interface PasswordEncoder
{
    /**
     * Encodes a password and returns it as a String suitable for storage by the client.
     * <p>
     * Implementations must perform a one-way hashing operation on the rawPassword
     * so that the rawPassword cannot practically be derived from the encoded result by
     * an attacker.
     * <p>
     * It is recommended that implementations include a unique prefix in their encoded form
     * which will allow {@link #canDecodePassword(String)} to be implemented easily.
     *
     * @param rawPassword the password provided by the user
     * @return the encoded password
     * @throws IllegalArgumentException if the rawPassword is null or empty
     */
    String encodePassword(String rawPassword) throws IllegalArgumentException;

    /**
     * Returns <tt>true</tt> if the rawPassword matches the stored password hash in
     * encodedPassword, otherwise <tt>false</tt>. The encodedPassword parameter should
     * be the result of an earlier call to {@link #encodePassword(String)}. If the encoded password
     * is not in a format which is handled by this encoder, this method will return <tt>false</tt>.
     * <p>
     * If multiple encodings are supported by an application, the client should call
     * {@link #canDecodePassword(String)} to check that the password was generated by
     * this encoder before calling this method.
     *
     * @param rawPassword the raw password provided by the user for authentication
     * @param encodedPassword the stored password associated with the user
     * @return true if the rawPassword is a match for the
     * @throws IllegalArgumentException if either rawPassword or encodedPassword is null or empty
     */
    boolean isValidPassword(String rawPassword, String encodedPassword) throws IllegalArgumentException;

    /**
     * Returns <tt>true</tt> if the encodedPassword is in the right format for decoding and
     * verification by this implementation, otherwise <tt>false</tt>. For example, implementations
     * might check the length of the encoded password or look for a particular prefix in the encoded
     * string.
     *
     * @param encodedPassword the stored password associated with this user
     * @return true if the encodedPassword can be decoded by this implementation, otherwise false
     */
    boolean canDecodePassword(String encodedPassword);
}
