package com.atlassian.scheduler.config;

import java.util.Date;

import javax.annotation.Nullable;
import javax.annotation.concurrent.Immutable;

import com.atlassian.annotations.PublicApi;

import static com.atlassian.scheduler.util.Safe.copy;
import static com.atlassian.util.concurrent.Assertions.isTrue;
import static com.google.common.base.Objects.equal;

/**
 * The description of an {@link Schedule#forInterval(long, Date)} interval schedule}.
 *
 * @since v1.0
 */
@Immutable
@PublicApi
public final class IntervalScheduleInfo
{
    private final Date firstRunTime;
    private final long intervalInMillis;

    IntervalScheduleInfo(@Nullable final Date firstRunTime, final long intervalInMillis)
    {
        this.firstRunTime = copy(firstRunTime);
        this.intervalInMillis = intervalInMillis;

        isTrue("intervalInMillis must not be negative", intervalInMillis >= 0L);
    }



    @Nullable
    public Date getFirstRunTime()
    {
        return copy(firstRunTime);
    }

    public long getIntervalInMillis()
    {
        return intervalInMillis;
    }



    @Override
    public boolean equals(@Nullable final Object o)
    {
        if (this == o)
        {
            return true;
        }
        if (o == null || getClass() != o.getClass())
        {
            return false;
        }

        final IntervalScheduleInfo other = (IntervalScheduleInfo) o;
        return intervalInMillis == other.intervalInMillis
                && equal(firstRunTime, other.firstRunTime);
    }

    @Override
    public int hashCode()
    {
        int result = firstRunTime != null ? firstRunTime.hashCode() : 0;
        result = 31 * result + (int)(intervalInMillis ^ (intervalInMillis >>> 32));
        return result;
    }

    @Override
    public String toString()
    {
        return "IntervalScheduleInfo[firstRunTime=" + firstRunTime + ",intervalInMillis=" + intervalInMillis + ']';
    }
}
