package com.atlassian.sal.core.net;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.Collections;
import java.util.Map;
import java.util.Optional;
import java.util.TreeMap;

import org.apache.hc.client5.http.impl.classic.CloseableHttpResponse;
import org.apache.hc.core5.http.ContentType;
import org.apache.hc.core5.http.EntityDetails;
import org.apache.hc.core5.http.Header;
import org.apache.hc.core5.http.io.entity.EntityUtils;

import com.atlassian.sal.api.net.Response;
import com.atlassian.sal.api.net.ResponseException;

public class HttpClientResponse implements Response {

    private final int statusCode;
    private final String statusText;
    private final byte[] body;
    private final Charset bodyCharset;
    private final Map<String, String> headers;

    public HttpClientResponse(final CloseableHttpResponse response) throws IOException {
        statusCode = response.getCode();
        statusText = response.getReasonPhrase();
        body = (response.getEntity() != null) ? EntityUtils.toByteArray(response.getEntity()) : new byte[0];
        this.bodyCharset = ContentType.getCharset(
                Optional.ofNullable(response.getEntity())
                        .map(EntityDetails::getContentType)
                        .map(ContentType::parse)
                        .orElse(null),
                // retain compatibility with the httpclient 4
                StandardCharsets.ISO_8859_1);
        final Map<String, String> extractedHeaders = new TreeMap<>(String.CASE_INSENSITIVE_ORDER);
        for (Header header : response.getHeaders()) {
            extractedHeaders.put(header.getName(), header.getValue());
        }
        headers = Collections.unmodifiableMap(extractedHeaders);
    }

    @Override
    public int getStatusCode() {
        return statusCode;
    }

    @Override
    public String getStatusText() {
        return statusText;
    }

    @Override
    public boolean isSuccessful() {
        return statusCode >= 200 && statusCode < 400;
    }

    @Override
    public String getHeader(final String name) {
        return headers.get(name);
    }

    @Override
    public Map<String, String> getHeaders() {
        return headers;
    }

    @Override
    public String getResponseBodyAsString() throws ResponseException {
        Charset decodeCharset = (bodyCharset != null) ? bodyCharset : Charset.defaultCharset();
        return new String(body, decodeCharset);
    }

    @Override
    public InputStream getResponseBodyAsStream() throws ResponseException {
        return new ByteArrayInputStream(body);
    }

    @Override
    public <T> T getEntity(final Class<T> entityClass) throws ResponseException {
        throw new UnsupportedOperationException("Not implemented");
    }
}
