package com.atlassian.sal.core.net;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;

import org.apache.hc.client5.http.ClientProtocolException;
import org.apache.hc.client5.http.auth.UsernamePasswordCredentials;
import org.apache.hc.client5.http.classic.methods.HttpGet;
import org.apache.hc.client5.http.classic.methods.HttpPost;
import org.apache.hc.client5.http.classic.methods.HttpPut;
import org.apache.hc.client5.http.config.RequestConfig;
import org.apache.hc.client5.http.cookie.StandardCookieSpec;
import org.apache.hc.client5.http.entity.UrlEncodedFormEntity;
import org.apache.hc.client5.http.entity.mime.MultipartEntityBuilder;
import org.apache.hc.client5.http.impl.auth.BasicScheme;
import org.apache.hc.client5.http.impl.classic.CloseableHttpClient;
import org.apache.hc.client5.http.impl.classic.CloseableHttpResponse;
import org.apache.hc.client5.http.protocol.HttpClientContext;
import org.apache.hc.core5.http.ClassicHttpRequest;
import org.apache.hc.core5.http.ContentType;
import org.apache.hc.core5.http.HttpHost;
import org.apache.hc.core5.http.NameValuePair;
import org.apache.hc.core5.http.io.entity.StringEntity;
import org.apache.hc.core5.http.io.support.ClassicRequestBuilder;
import org.apache.hc.core5.http.message.BasicNameValuePair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;

import com.atlassian.sal.api.net.Request;
import com.atlassian.sal.api.net.RequestFilePart;
import com.atlassian.sal.api.net.Response;
import com.atlassian.sal.api.net.ResponseException;
import com.atlassian.sal.api.net.ResponseHandler;
import com.atlassian.sal.api.net.ResponseProtocolException;
import com.atlassian.sal.api.net.ResponseStatusException;
import com.atlassian.sal.api.net.ReturningResponseHandler;
import com.atlassian.util.profiling.Ticker;

import static java.util.Objects.requireNonNull;

import static com.atlassian.plugin.util.PluginKeyStack.getFirstPluginKey;
import static com.atlassian.util.profiling.Metrics.metric;

/**
 * HttpClient implementation of Request interface
 */
public class HttpClientRequest<T extends Request<?, ?>, RESP extends Response>
        implements Request<HttpClientRequest<?, ?>, HttpClientResponse> {
    @VisibleForTesting
    static final String METRIC_NAME = "http.sal.request";

    private static final Logger log = LoggerFactory.getLogger(HttpClientRequest.class);

    private final CloseableHttpClient httpClient;
    private final List<NameValuePair> requestParameters;
    protected final HttpClientContext httpClientContext;
    final ClassicRequestBuilder requestBuilder;
    final RequestConfig.Builder requestConfigBuilder;

    // Unfortunately we need to keep a list of the headers
    private final Map<String, List<String>> headers = new HashMap<>();
    private final String pluginKey;

    public HttpClientRequest(
            CloseableHttpClient httpClient,
            HttpClientContext httpClientContext,
            MethodType initialMethodType,
            String initialUrl) {
        this.httpClient = httpClient;
        this.httpClientContext = httpClientContext;
        this.requestBuilder =
                ClassicRequestBuilder.create(initialMethodType.toString()).setUri(initialUrl);
        this.requestParameters = new LinkedList<>();
        pluginKey = getFirstPluginKey();

        final ConnectionConfig connectionConfig = new SystemPropertiesConnectionConfig();
        this.requestConfigBuilder = RequestConfig.custom()
                .setConnectTimeout(connectionConfig.getConnectionTimeout(), TimeUnit.MILLISECONDS)
                .setResponseTimeout(connectionConfig.getSocketTimeout(), TimeUnit.MILLISECONDS)
                .setMaxRedirects(connectionConfig.getMaxRedirects())
                .setCookieSpec(StandardCookieSpec.RELAXED);
    }

    @Override
    public String execute() throws ResponseException {
        return executeAndReturn(response -> {
            if (!response.isSuccessful()) {
                throw new ResponseStatusException(
                        "Unexpected response received. Status code: " + response.getStatusCode(), response);
            }
            return response.getResponseBodyAsString();
        });
    }

    @Override
    public void execute(final ResponseHandler<? super HttpClientResponse> responseHandler) throws ResponseException {
        executeAndReturn((ReturningResponseHandler<HttpClientResponse, Void>) response -> {
            responseHandler.handle(response);
            return null;
        });
    }

    @Override
    public <RET> RET executeAndReturn(final ReturningResponseHandler<? super HttpClientResponse, RET> responseHandler)
            throws ResponseException {
        if (!requestParameters.isEmpty()) {
            requestBuilder.setEntity(new UrlEncodedFormEntity(requestParameters, StandardCharsets.UTF_8));
        }
        httpClientContext.setRequestConfig(requestConfigBuilder.build());
        final ClassicHttpRequest request = requestBuilder.build();
        log.debug("Executing request:{}", request);

        try (final CloseableHttpResponse response = httpClient.execute(request, httpClientContext)) {
            try (Ticker ignored = metric(METRIC_NAME)
                    .tag("pluginKeyAtCreation", pluginKey)
                    .tag("action", request.getMethod())
                    .optionalTag("url", request.getRequestUri())
                    .withAnalytics()
                    .startTimer()) {
                return responseHandler.handle(new HttpClientResponse(response));
            }
        } catch (ClientProtocolException cpe) {
            throw new ResponseProtocolException(cpe);
        } catch (IOException e) {
            throw new ResponseException(e);
        }
    }

    @Override
    public Map<String, List<String>> getHeaders() {
        return Collections.unmodifiableMap(headers);
    }

    @Override
    public HttpClientRequest addBasicAuthentication(
            final String hostname, final String username, final String password) {
        BasicScheme authScheme = new BasicScheme();
        authScheme.initPreemptive(new UsernamePasswordCredentials(username, password.toCharArray()));
        httpClientContext.getAuthCache().put(new HttpHost(hostname), authScheme);
        return this;
    }

    @Override
    public HttpClientRequest setConnectionTimeout(final int connectionTimeout) {
        requestConfigBuilder.setConnectionRequestTimeout(connectionTimeout, TimeUnit.MILLISECONDS);
        return this;
    }

    @Override
    public HttpClientRequest setSoTimeout(final int soTimeout) {
        requestConfigBuilder.setResponseTimeout(soTimeout, TimeUnit.MILLISECONDS);
        return this;
    }

    @Override
    public HttpClientRequest setUrl(final String url) {
        requestBuilder.setUri(url);
        return this;
    }

    @Override
    public HttpClientRequest setRequestBody(final String requestBody) {
        return setRequestBody(requestBody, ContentType.TEXT_PLAIN.getMimeType());
    }

    @Override
    public HttpClientRequest setRequestBody(final String requestBodyString, final String contentTypeString) {
        requireNonNull(requestBodyString);
        requireNonNull(contentTypeString);
        Preconditions.checkState(isRequestBodyMethod(), "Only PUT or POST methods accept a request body.");

        requestBuilder.setEntity(
                new StringEntity(requestBodyString, ContentType.create(contentTypeString, StandardCharsets.UTF_8)));
        return this;
    }

    @Override
    public HttpClientRequest setFiles(final List<RequestFilePart> requestBodyFiles) {
        requireNonNull(requestBodyFiles);
        Preconditions.checkState(isRequestBodyMethod(), "Only PUT or POST methods accept a request body.");

        final MultipartEntityBuilder multipartEntityBuilder = MultipartEntityBuilder.create();

        for (RequestFilePart requestBodyFile : requestBodyFiles) {
            final ContentType fileContentType = ContentType.create(requestBodyFile.getContentType());
            multipartEntityBuilder.addBinaryBody(
                    requestBodyFile.getParameterName(),
                    requestBodyFile.getFile(),
                    fileContentType,
                    requestBodyFile.getFileName());
        }

        requestBuilder.setEntity(multipartEntityBuilder.build());
        return this;
    }

    @Override
    public HttpClientRequest addRequestParameters(final String... params) {
        requireNonNull(params);
        Preconditions.checkState(isRequestBodyMethod(), "Only PUT or POST methods accept a request body.");

        if (params.length % 2 != 0) {
            throw new IllegalArgumentException("You must enter an even number of arguments.");
        }

        for (int i = 0; i < params.length; i += 2) {
            final String name = params[i];
            final String value = params[i + 1];
            requestParameters.add(new BasicNameValuePair(name, value));
        }

        return this;
    }

    private boolean isRequestBodyMethod() {
        final String methodType = requestBuilder.getMethod();
        return HttpPost.METHOD_NAME.equals(methodType)
                || HttpPut.METHOD_NAME.equals(methodType)
                || HttpGet.METHOD_NAME.equals(methodType);
    }

    @Override
    public HttpClientRequest addHeader(final String headerName, final String headerValue) {
        headers.computeIfAbsent(headerName, k -> new ArrayList<>()).add(headerValue);
        requestBuilder.addHeader(headerName, headerValue);
        return this;
    }

    @Override
    public HttpClientRequest setHeader(final String headerName, final String headerValue) {
        headers.put(headerName, new ArrayList<>(Collections.singletonList(headerValue)));
        requestBuilder.setHeader(headerName, headerValue);
        return this;
    }

    @Override
    public HttpClientRequest setFollowRedirects(final boolean follow) {
        requestConfigBuilder.setRedirectsEnabled(follow);
        return this;
    }

    @Override
    public HttpClientRequest setEntity(final Object entity) {
        throw new UnsupportedOperationException(
                "This SAL request does not support object marshalling. Use the RequestFactory component instead.");
    }
}
