package com.atlassian.sal.core.net;

import java.io.IOException;
import java.net.InetAddress;
import java.net.Socket;
import java.net.UnknownHostException;
import java.util.Arrays;

import javax.net.ssl.SSLSocket;

import org.apache.commons.httpclient.ConnectTimeoutException;
import org.apache.commons.httpclient.params.HttpConnectionParams;
import org.apache.commons.httpclient.protocol.SSLProtocolSocketFactory;

/**
 * A custom SecureProtocolSocketFactory that uses JSSE to create sockets.
 * <p/>
 * This factory handles the JSSE system property "https.protocols" used to specify which protocol suites to enable.
 * http://docs.oracle.com/javase/1.5.0/docs/guide/security/jsse/JSSERefGuide.html#Customization
 */
class CustomSSLProtocolSocketFactory extends SSLProtocolSocketFactory
{
    public static final String DEFAULT_PROTOCOLS = "TLSv1,SSLv3";
    private final String[] protocols;

    /**
     * Creates CustomSSLProtocolSocketFactory.
     * Uses the "https.protocols" system property if defined otherwise uses protocols parameter.
     *
     * Available protocols:
     * http://docs.oracle.com/javase/1.5.0/docs/guide/security/jsse/JSSERefGuide.html#AppA
     *
     * @param protocols Comma-separated list of protocol versions enabled for use on socket created by this factory.
     */
    public CustomSSLProtocolSocketFactory(String protocols)
    {
        String protocolsProperty = System.getProperty("https.protocols", protocols);
        if (protocolsProperty == null)
        {
            protocolsProperty = DEFAULT_PROTOCOLS;
        }

        this.protocols = protocolsProperty.split(",");
    }

    private void setSocketProtocols(Socket socket)
    {
        SSLSocket sslSocket = (SSLSocket) socket;
        sslSocket.setEnabledProtocols(protocols);
    }

    @Override
    public Socket createSocket(final String host, final int port, final InetAddress clientHost, final int clientPort)
            throws IOException, UnknownHostException
    {
        Socket socket = super.createSocket(host, port, clientHost, clientPort);
        setSocketProtocols(socket);
        return socket;
    }

    @Override
    public Socket createSocket(final String host, final int port, final InetAddress localAddress, final int localPort, final HttpConnectionParams params)
            throws IOException, UnknownHostException, ConnectTimeoutException
    {
        Socket socket = super.createSocket(host, port, localAddress, localPort, params);
        setSocketProtocols(socket);
        return socket;
    }

    @Override
    public Socket createSocket(final String host, final int port) throws IOException, UnknownHostException
    {
        Socket socket = super.createSocket(host, port);
        setSocketProtocols(socket);
        return socket;
    }

    @Override
    public Socket createSocket(final Socket socket, final String host, final int port, final boolean autoClose)
            throws IOException, UnknownHostException
    {
        Socket newSocket = super.createSocket(socket, host, port, autoClose);
        setSocketProtocols(newSocket);
        return newSocket;
    }

    public String[] getProtocols()
    {
        return protocols;
    }

    @Override
    public boolean equals(Object o)
    {
        if (this == o)
        {
            return true;
        }
        if (o == null || getClass() != o.getClass())
        {
            return false;
        }
        if (!super.equals(o))
        {
            return false;
        }

        CustomSSLProtocolSocketFactory that = (CustomSSLProtocolSocketFactory) o;
        if (!Arrays.equals(protocols, that.protocols))
        {
            return false;
        }

        return true;
    }

    @Override
    public int hashCode()
    {
        int result = super.hashCode();
        result = 31 * result + (protocols != null ? Arrays.hashCode(protocols) : 0);
        return result;
    }
}
