package com.atlassian.sal.api.validate;

import com.atlassian.annotations.PublicApi;
import com.google.common.collect.ImmutableSet;

import javax.annotation.Nonnull;
import java.util.Collections;

/**
 * The outcome of a license validation check for each product key.
 *
 * @since 4.1
 */
@PublicApi
public class LicenseValidationResult {

    private final String productKey;

    private final ImmutableSet<LicenseValidationError> errorMessages;

    private final ImmutableSet<LicenseValidationWarning> warningMessages;

    private LicenseValidationResult(final String productKey, final Iterable<LicenseValidationError> errorMessages, final
    Iterable<LicenseValidationWarning> warningMessages) {
        this.productKey = productKey;
        this.errorMessages = ImmutableSet.copyOf(errorMessages);
        this.warningMessages = ImmutableSet.copyOf(warningMessages);
    }

    public static LicenseValidationResult withErrorMessages(@Nonnull final String productKey, @Nonnull final Iterable<LicenseValidationError> errorMessages) {
        return new LicenseValidationResult(productKey, errorMessages, Collections.emptySet());
    }

    public static LicenseValidationResult withWarningMessages(@Nonnull final String productKey, @Nonnull final Iterable<LicenseValidationWarning> warningMessages) {
        return new LicenseValidationResult(productKey, Collections.emptySet(), warningMessages);
    }

    public static LicenseValidationResult withErrorAndWarningMessages(@Nonnull final String productKey, @Nonnull final Iterable<LicenseValidationError> errorMessages, @Nonnull final Iterable<LicenseValidationWarning> warningMessages) {
        return new LicenseValidationResult(productKey, errorMessages, warningMessages);
    }

    /**
     * Return true if the validation passed, false if there were errors. (Note warnings do not cause this method to
     * return false).
     *
     * @return true if the validation passed, false if there were errors.
     * @see #hasWarnings()
     */
    public boolean isValid() {
        return errorMessages.isEmpty();
    }

    /**
     * Return true if the validation added error messages.
     *
     * @return true if the validation failed with error messages.
     */
    public boolean hasErrors() {
        return !errorMessages.isEmpty();
    }

    /**
     * Return true if the validation added warning messages.
     *
     * @return true if the validation failed with warning messages.
     */
    public boolean hasWarnings() {
        return !warningMessages.isEmpty();
    }

    /**
     * Returns validation error for each products.
     * <p>
     * This will never be null, but may be empty.
     *
     * @return validation error messages, could be empty but not null.
     */
    @Nonnull
    public Iterable<LicenseValidationError> getErrorMessages() {
        return errorMessages;
    }

    /**
     * Returns validation warnings for each products.
     * <p>
     * This will never be null, but may be empty.
     *
     * @return validation warning messages, could be empty but not null.
     */
    @Nonnull
    public Iterable<LicenseValidationWarning> getWarningMessages() {
        return warningMessages;
    }
}
