package com.atlassian.sal.api.license;

import com.atlassian.annotations.PublicApi;
import com.atlassian.sal.api.validate.ValidationResult;

import java.util.Collection;
import java.util.List;
import java.util.Locale;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;

/**
 * Interface into the license system for the underlying application.
 * <p>
 *     Most applications will take a single global license for the entire application, but some are actually a platform
 *     with multiple "products" installed each with its own product license.
 * </p>
 *
 * @since 2.0
 */
@SuppressWarnings ("UnusedDeclaration")
@PublicApi
public interface LicenseHandler
{
    /**
     * Sets the license string for the currently running application.
     *
     * <p>
     * This method is not suitable for platforms.
     *
     * @param license The raw license string
     *
     * @throws IllegalArgumentException if the license string is not a valid license
     * @throws java.lang.UnsupportedOperationException if this is a platform that allows multiple licenses
     *
     * @deprecated Use {@link #addProductLicense(String, String)} instead. Since 3.0.
     */
    void setLicense(String license);

    /**
     * Use this to figure out if this host application uses a single global license, or if it is a platform that can take
     * multiple Product licenses.
     * <p>
     * Most applications would return false here but Fisheye/Crucible will return true, and JIRA 7.0 will return true.
     *
     * @return {@code true} if this application is a platform that accepts multiple product licenses, {@code false} if it just takes a single global license.
     * @since 3.0
     * @see #hostAllowsCustomProducts()
     */
    boolean hostAllowsMultipleLicenses();

    /**
     * Returns true if it is possible to add custom products on top of this platform.
     * <p>
     * Most applications would return false here.
     * Confluence returns true because it has Spaces and Questions, JIRA 7.0 will return true, but Fisheye/Crucible will
     * return false - although it has separate licenses for FishEye and Crucible, you cannot add new custom products.
     *
     * @return {@code true} if this application is a platform that accepts multiple product licenses, {@code false} if it just takes a single global license.
     * @since 3.0
     * @see #hostAllowsMultipleLicenses()
     */
    boolean hostAllowsCustomProducts();

    /**
     * Returns the list of currently available products in this host application whether these products are currently licensed or not.
     *
     * <p>
     * For FishEye/Crucible this will return both "fisheye" and "crucible".
     * <br>
     * For JIRA, it will return the list of products that are currently installed (eg "com.atlassian.servicedesk").
     * <br>
     * For simple applications that only take a single license, it will return a single application name where that
     * name is the ID used by HAMS to define the application in the license eg "bamboo", "conf".
     * <br>
     *
     * @return the list of currently available products in this host application
     */
    List<String> getProductKeys();

    /**
     * Adds the given product license to the host application.
     *
     * <p>
     *     For a platform that can take multiple license, the platform will figure out which product this is for and
     *     replace the current license for that product if one exists.
     * </p>
     *
     * @param productKey The product to add this license to
     * @param license The license string
     *
     * @throws java.lang.IllegalArgumentException if the license string is not a valid license for the given productKey
     * @since 3.0
     */
    void addProductLicense(@Nonnull String productKey, @Nonnull String license);

    /**
     * Validates that the given license is valid to add for the given product.
     *
     * @param productKey The product to add this license to
     * @param license the raw license String
     * @param locale locale of the end user - this is used to internationalise the error messages if any.
     */
    @Nonnull
    ValidationResult validateProductLicense(@Nonnull String productKey, @Nonnull String license, @Nullable Locale locale);

    /**
     * Gets the server ID of the currently running application.  The server ID format is four quadruples of
     * alphanumeric characters, each separated by a dash (<tt>-</tt>).
     *
     * @return the server ID, or {@code null} if the server ID has not yet
     *         been set for the currently running application.
     *
     * @since 2.7
     */
    @Nullable
    String getServerId();

    /**
     * Gets the Support Entitlement Number (SEN) for the single license in the currently running application.
     * <p>
     * This method is not suitable for platforms because these may have multiple licenses and hence multiple SENs.
     *
     * @return the Support Entitlement Number, or {@code null} if there is no current support entitlement.
     *
     * @throws java.lang.UnsupportedOperationException if this is a platform that allows multiple licenses
     * @since 2.7
     *
     * @deprecated Get the license details and call {@link BaseLicenseDetails#getSupportEntitlementNumber()}. Since 3.0.
     */
    @Nullable
    String getSupportEntitlementNumber();

    /**
     * Returns the raw license String for a given product on a platform.
     *
     * @param productKey the product key.
     * @return the raw license String for the given product key.
     *
     * @see #getProductLicenseDetails(String)
     * @since 3.0
     */
    @Nullable
    String getRawProductLicense(String productKey);

    /**
     * Returns the license details for a given individual product.
     *
     * @param productKey the product key.
     * @return the license details for the given product key.
     *
     * @see #getRawProductLicense(String)
     * @see #decodeLicenseDetails(String)
     * @see #getAllProductLicenses()
     * @since 3.0
     */
    @Nullable
    SingleProductLicenseDetailsView getProductLicenseDetails(@Nonnull String productKey);

    /**
     * Returns all the product licenses on a platform.
     * <p>
     *     A platform may return multiple licenses, a simple application will return 0 or 1.
     * </p>
     *
     * @return all the product licenses on a platform.
     *
     * @see #getProductLicenseDetails(String)
     * @since 3.0
     */
    @Nonnull
    Collection<MultiProductLicenseDetails> getAllProductLicenses();

    /**
     * Decodes a Platform Product License from the raw license String.
     * This can be used to validate an entered license String before attempting to save it to the application.
     *
     * @param license the raw license String
     * @return the license details for the given product key.
     *
     * @throws java.lang.IllegalArgumentException if the license string is not able to be decoded in this host application
     * @see #getProductLicenseDetails(String)
     * @since 3.0
     */
    @Nonnull
    MultiProductLicenseDetails decodeLicenseDetails(@Nonnull String license);
}
