package com.atlassian.sal.api.usersettings;

import com.google.common.base.Function;

/**
 * Service for getting and updating immutable UserSettings objects stored against a user name
 *
 * UserSettings can be used to store values of type String, Boolean and Long - nothing more.
 * Max key length is 255, Values of type String also have a max length of 255.
 *
 * @since 2.9
 */
public interface UserSettingsService
{
    /**
     * Gets the UserSettings for the user with name userName
     * @param userName name of the user whose user settings are to be retrieved
     * @return a UserSettings for the user with name userName,
     * @throws IllegalArgumentException if no user could be found with that name
     */
    UserSettings getUserSettings(String userName);

    /**
     * Updates the UserSettings stored for this user with name UserName. Implementations of this interface will ensure
     * that updateFunctions provided to this method are called in a threadsafe manner.
     *
     * Consumers can throw RuntimeExceptions within updateFunction to control flow when the input to updateFunction
     * is unexpected. As such, implementers must either rethrow caught RuntimeExceptions, or not catch them in the first place.
     *
     * @param userName name of the user whose UserSettings are to be updated. If userName does not match a known user,
     *                 updateFunction will not be called.
     * @param updateFunction function to update a user's UserSettings. The parameter to updateFunction is a
     *                       UserSettingsBuilder whose contents match the UserSettings for the provided user. The
     *                       return value of updateFunction will be stored against the specified user.
     *
     * @throws IllegalArgumentException if no user could be found with that name
     * @throws UnsupportedOperationException if updateFunction creates entries with key length > 255 or with a String value
     * with length > 255
     */
    void updateUserSettings(String userName, Function<UserSettingsBuilder, UserSettings> updateFunction);

}
