package com.atlassian.sal.api.features;

import com.atlassian.fugue.Option;
import com.atlassian.sal.api.user.UserResolutionException;

import javax.annotation.Nullable;

/**
 * Provides a cross-product method for determining whether a dark feature is enabled.
 *
 * Implementing products can back these checks with their own internal dark feature management system, but must follow
 * the enable and disable dark features on startup based on system properties and the contents of an optional properties
 * file.
 *
 * Dark feature keys must begin with atlassian.darkfeature. Values must be either true or false. Location of dark
 * features property file can be overridden with darkfeatures.properties.file system property.
 *
 * See SystemDarkFeatureInitializer in sal-core for implementation.
 *
 * @since 2.10
 */
public interface DarkFeatureManager
{
    /**
     * Prefix for all dark feature property keys.
     */
    public static final String ATLASSIAN_DARKFEATURE_PREFIX = "atlassian.darkfeature.";

    /**
     * System property for disabling all dark features.
     */
    public static final String DISABLE_ALL_DARKFEATURES_PROPERTY = "atlassian.darkfeature.disabled";

    /**
     * System property for overriding location of dark features property file.
     */
    public static final String DARKFEATURES_PROPERTIES_FILE_PROPERTY = "darkfeatures.properties.file";

    /**
     * Default properties file name.
     */
    public static final String DARKFEATURES_PROPERTIES_FILE_PROPERTY_DEFAULT = "atlassian-darkfeatures.properties";

    /**
     * Checks if a dark feature is enabled for all users, regardless whether the feature can be changed during runtime
     * or not.
     * @param featureKey key of the feature to be checked
     * @return true if the feature key is valid and enabled, false otherwise
     * @see ValidFeatureKeyPredicate
     */
    boolean isFeatureEnabledForAllUsers(String featureKey);

    /**
     * Checks if a dark feature is enabled for the current user (must be called within the context of a request). Any
     * issues while should result in a <code>false</code> return value instead of an exception.
     * Returns <code>true</code> if the feature is enabled for all users or the current user only.
     * @param featureKey key of the feature to be checked
     * @return true if the feature key is valid and enabled, false otherwise
     * @see ValidFeatureKeyPredicate
     */
    boolean isFeatureEnabledForCurrentUser(String featureKey);

    /**
     * Checks if a dark feature is enabled for all users or just for the user with name username. In case the user is
     * anonymous, only features enabled for all users are included.
     * @param username the name of the user being queried; <code>null</code> represents the anonymous user
     * @param featureKey key of the feature to be checked
     * @return an Option.Some containing Boolean.TRUE if the feature key is valid and enabled for the user,
     * containing Boolean.FALSE otherwise. If no user exists with name username then returns Option.None
     * @see ValidFeatureKeyPredicate
     */
    Option<Boolean> isFeatureEnabledForUser(@Nullable String username, String featureKey);

    /**
     * Returns true if the current acting user has permission to change dark features for all users. This is a nothrow
     * method and should return a value instead of throw an exception.
     * @return <code>true</code> iff the current acting user has permission to change dark features for all users,
     * <code>false</code> otherwise
     */
    boolean canManageFeaturesForAllUsers();

    /**
     * Enable the given dark feature all users. The acting user must have permission to change dark features for all
     * users.
     * @param featureKey key of the feature to be enabled
     * @throws InvalidFeatureKeyException if the feature key is not valid
     * @throws MissingPermissionException if the user has not the required permission
     * @throws IllegalStateException if the update failed
     * @see ValidFeatureKeyPredicate
     * @see #canManageFeaturesForAllUsers()
     */
    void enableFeatureForAllUsers(String featureKey);

    /**
     * Disable the given dark feature for all users. The acting user must have permission to change dark features for
     * all users.
     * @param featureKey key of the feature to be disabled
     * @throws InvalidFeatureKeyException if the feature key is not valid
     * @throws MissingPermissionException if the user has not the required permission
     * @throws IllegalStateException if the update failed
     * @see ValidFeatureKeyPredicate
     * @see #canManageFeaturesForAllUsers()
     */
    void disableFeatureForAllUsers(String featureKey);

    /**
     * Enable a dark feature for the current user only. Anonymous users are not supported. <strong>If the feature is
     * already enabled for all users, the user will still be able to use it.</strong>
     * @param featureKey key of the feature to enable
     * @throws InvalidFeatureKeyException if the feature key is not valid
     * @throws UserResolutionException if the current user could not be resolved or is anonymous
     * @throws IllegalStateException if the update failed
     * @see ValidFeatureKeyPredicate
     */
    void enableFeatureForCurrentUser(String featureKey);

    /**
     * Enable a dark feature for the given user only. Anonymous users are not supported. <strong>If the feature is
     * already enabled for all users, the user will still be able to use it.</strong>
     * @param username name of the user to enable the feature for; not <code>null</code>
     * @param featureKey key of the feature to be enabled
     * @throws InvalidFeatureKeyException if the feature key is not valid
     * @throws UserResolutionException if the username to user resolution failed
     * @throws IllegalStateException if the update failed
     * @see ValidFeatureKeyPredicate
     */
    void enableFeatureForUser(String username, String featureKey);

    /**
     * Disable a dark feature for the current user only. Anonymous users are not supported. <strong>If the feature is
     * enabled system or site wide the user will still be able to use it.</strong>
     * @param featureKey key of the feature to be disabled
     * @throws InvalidFeatureKeyException if the feature key is not valid
     * @throws UserResolutionException if the current user could not be resolved or is anonymous
     * @throws IllegalStateException if the update failed
     * @see ValidFeatureKeyPredicate
     */
    void disableFeatureForCurrentUser(String featureKey);

    /**
     * Disable a dark feature for the given user only. Anonymous users are not supported. <strong>If the feature is
     * enabled for all users, the user will still be able to use it.</strong>
     * @param username name of the user to disable the feature for; not <code>null</code>
     * @param featureKey key of the feature to be disabled
     * @throws InvalidFeatureKeyException if the feature key is not valid
     * @throws UserResolutionException if the username to user resolution failed
     * @throws IllegalStateException if the update failed
     * @see ValidFeatureKeyPredicate
     */
    void disableFeatureForUser(String username, String featureKey);

    /**
     * @return features enabled for all user.
     */
    EnabledDarkFeatures getFeaturesEnabledForAllUsers();

    /**
     * Return features enabled for the current user (must be called within the context of a request). In case the
     * current user is anonymous, all global enabled features are returned.
     * @return all dark features applicable for the current user.
     */
    EnabledDarkFeatures getFeaturesEnabledForCurrentUser();

    /**
     * Return enabled features for a given user. Avoids throwing exceptions by returning
     * {@link com.atlassian.fugue.Option#none()} if there are any issues resolving the user. In case the current user is
     * anonymous, all global enabled features are returned.
     * @param username the name of the user being queried; <code>null</code> represents the anonymous user
     * @return all dark features applicable for the given user or {@link com.atlassian.fugue.Option#none()} if no user
     * with name username exists.
     */
    Option<EnabledDarkFeatures> getFeaturesEnabledForUser(@Nullable String username);

}
