package com.atlassian.sal.api.features;

import com.google.common.base.Predicate;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Maps;

import javax.annotation.Nullable;
import javax.annotation.concurrent.Immutable;
import java.util.Collections;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;

import static com.google.common.base.Preconditions.checkNotNull;

/**
 * Represents a set of enabled features.
 *
 * @since 2.10
 */
@Immutable
public final class EnabledDarkFeatures
{
    /**
     * Shorthand in case there are no enabled dark features or they are disabled
     */
    public static final EnabledDarkFeatures NONE = new EnabledDarkFeatures(Collections.<String, FeatureKeyClassifier>emptyMap());

    private final ImmutableMap<String, FeatureKeyClassifier> enabledFeatures;

    /**
     * Create an instance of enabled features based on the given map.
     * @param enabledFeatures the map of enabled dark features
     */
    public EnabledDarkFeatures(final Map<String, FeatureKeyClassifier> enabledFeatures)
    {
        this.enabledFeatures = ImmutableMap.copyOf(enabledFeatures);
    }

    /**
     * Create an instance with the specified features enabled for all users.
     * @param systemEnabledFeatures the system wide enabled features
     * @param siteEnabledFeatures the site wide enabled features
     * @return an instance containing the information in one class
     */
    public static EnabledDarkFeatures createFromSet(@Nullable final Set<String> systemEnabledFeatures, @Nullable final Set<String> siteEnabledFeatures)
    {
        return createFromSet(systemEnabledFeatures, siteEnabledFeatures, Collections.<String>emptySet());
    }

    /**
     * Create an instance with the specified features enabled for the current user.
     * @param systemEnabledFeatures the system wide enabled features
     * @param siteEnabledFeatures the site wide enabled features
     * @param userEnabledFeatures the features enabled for the current user only
     * @return an instance containing the information in one class
     */
    public static EnabledDarkFeatures createFromSet(@Nullable final Set<String> systemEnabledFeatures, @Nullable final Set<String> siteEnabledFeatures, @Nullable final Set<String> userEnabledFeatures)
    {
        final Map<String, FeatureKeyClassifier> enabledFeatures =  ImmutableMap.<String, FeatureKeyClassifier>builder()
                .putAll(toMap(systemEnabledFeatures, FeatureKeyClassifier.ALL_USERS_READ_ONLY))
                .putAll(toMap(siteEnabledFeatures, FeatureKeyClassifier.ALL_USERS))
                .putAll(toMap(userEnabledFeatures, FeatureKeyClassifier.CURRENT_USER_ONLY))
                .build();
        return new EnabledDarkFeatures(enabledFeatures);
    }

    private static Map<? extends String, ? extends FeatureKeyClassifier> toMap(@Nullable final Set<String> enabledFeatures, final FeatureKeyClassifier classifier)
    {
        if (enabledFeatures == null)
        {
            return Collections.emptyMap();
        }

        final Map<String, FeatureKeyClassifier> map = new ConcurrentHashMap<String, FeatureKeyClassifier>();
        for (final String featureKey : enabledFeatures)
        {
            map.put(featureKey, classifier);
        }
        return map;
    }

    /**
     * Return all enabled feature keys.
     * @return all enabled feature keys
     */
    public ImmutableSet<String> getFeatureKeys()
    {
        return ImmutableSet.copyOf(enabledFeatures.keySet());
    }

    /**
     * Returns all enabled feature keys matching the given criteria.
     * @param criteria the filter condition to be applied on the set of enabled features
     * @return the filtered set of enabled features
     */
    public ImmutableSet<String> getFeatureKeys(final Predicate<FeatureKeyClassifier> criteria)
    {
        checkNotNull(criteria, "criteria");
        return ImmutableSet.copyOf(Maps.filterEntries(enabledFeatures, new Predicate<Map.Entry<String, FeatureKeyClassifier>>()
        {
            @Override
            public boolean apply(@Nullable final Map.Entry<String, FeatureKeyClassifier> input)
            {
                return input != null && criteria.apply(input.getValue());
            }
        }).keySet());
    }

    /**
     * Check that the given feature is enabled
     * @param featureKey the feature key to be checked
     * @return <code>true</code> if the given feature key is enabled, <code>false</code> otherwise
     */
    public boolean isFeatureEnabled(final String featureKey)
    {
        return enabledFeatures.containsKey(featureKey);
    }

    @Override
    public String toString()
    {
        return "EnabledDarkFeatures{enabledFeatures=" + enabledFeatures + '}';
    }
}
