package com.atlassian.renderer.wysiwyg.converter;

import com.atlassian.renderer.v2.RenderUtils;
import com.atlassian.renderer.wysiwyg.ListContext;
import com.atlassian.renderer.wysiwyg.NodeContext;
import org.apache.commons.lang.StringUtils;

/**
 * Converts an {@literal <li>} tag to a single list item in wiki markup based on the current {@link ListContext}.
 */
final class ListItemConverter implements Converter {
    static final ListItemConverter INSTANCE = new ListItemConverter();

    private ListItemConverter() {
    }

    public boolean canConvert(NodeContext nodeContext) {
        return nodeContext.hasNodeName("li");
    }

    public String convertNode(NodeContext nodeContext, DefaultWysiwygConverter wysiwygConverter) {
        NodeContext childConversionContext = new NodeContext.Builder(nodeContext)
                .inListItem(true).ignoreText(false).previousSibling(null).build();
        String convertedChildren = wysiwygConverter.convertChildren(childConversionContext);
        String itemContent = RenderUtils.trimInitialNewline(convertedChildren);
        // if the list item was empty, or contained just a <br>, put an &nbsp; placeholder in it
        if (itemContent.equals("") || itemContent.trim().equals("\\\\")) {
            itemContent = "&nbsp;";
        }
        // fix for CONF-5210, problem 2, <li> with a <br/> at the end
        while (itemContent.endsWith("\n")) {
            itemContent = StringUtils.chomp(itemContent);
        }
        // fix for CONF-5492, <li> with a <br> at the end which becomes "\\\\\n "
        while (itemContent.endsWith("\\\\\n ")) {
            itemContent = StringUtils.chomp(itemContent, "\\\\\n ");
        }
        // if the only contents of this list item is a nested list, don't create a line for it
        if (itemContent.trim().startsWith("##") || itemContent.trim().startsWith("**")) {
            return wysiwygConverter.getSeparator("li", nodeContext) + itemContent;
        }

        // RNDR-114 - In certain situations the items gets created without the surrounding list elements
        ListContext listContext = nodeContext.getListContext();
        if (listContext == null) {
            listContext = new ListContext(ListContext.BULLETED);
        }

        return wysiwygConverter.getSeparator("li", nodeContext) + listContext.decorateText(itemContent);
    }
}
