package com.atlassian.renderer.v2.plugin;

import com.atlassian.plugin.Plugin;
import com.atlassian.plugin.PluginParseException;
import com.atlassian.plugin.descriptors.AbstractModuleDescriptor;
import com.atlassian.plugin.web.descriptors.WeightedDescriptor;
import com.atlassian.renderer.v2.components.RendererComponent;
import org.dom4j.Element;

/**
 * A module descriptor for pluggable renderer components. Renderer component modules actually configure a
 * RendererComponentFactory, rather than returning the component directly, to allow for more flexible
 * configuration of different component types.
 * <p>
 * <p>Components are weighted, and are executed from lowest weight to highest.
 * <p>
 * <p>For simple components, you can instantiate the component directly:
 * <p>
 * <p><code>&lt;renderer-component key="mycomponent" weight="10" class="com.example.MyComponent"></code>
 * <p>
 * <p>For more complex components, you can defer component creation by specifying an instance of
 * RendererComponentFactory as the module class. Any standard module parameter will be included in
 * the factory's create method's parameter map.
 * <p>
 * <pre>&lt;renderer-component key="mycomponent" weight="20" class="com.example.InlineComponentFactory>
 *    &lt;param name="style">emphasis&lt;/param>
 *    &lt;param name="markup">_&lt;/param>
 * &;t;/renderer-component>
 *
 * <p>Applications may want to override the <code>instantiateModuleClass</code> method on this class to create
 * factories via their component manager.
 */
public class RendererComponentModuleDescriptor extends AbstractModuleDescriptor<RendererComponent> implements WeightedDescriptor {
    private RendererComponent component;
    private int weight;

    public void init(Plugin plugin, Element element) throws PluginParseException {
        super.init(plugin, element);
        initialiseWeight(element);
    }

    public void enabled() {
        super.enabled();
        assertComponentOfExpectedClass();
    }

    public RendererComponent getModule() {
        if (component == null)
            component = createComponent();

        return component;
    }

    public int getWeight() {
        return weight;
    }

    /**
     * Create a new instance of the module class. Applications will probably want to override this method to
     * perform instantiation through their component manager of choice.
     *
     * @return a new instance of the module class
     * @see #getModuleClass()
     */
    protected Object instantiateComponentClass() {
        try {
            return getModuleClass().newInstance();
        } catch (InstantiationException e) {
            throw new RuntimeException("Could not create component: " + getClass() + ": " + e, e);
        } catch (IllegalAccessException e) {
            throw new RuntimeException("Could not create component: " + getClass() + ": " + e, e);
        }
    }

    private RendererComponent createComponent() {
        Object o = instantiateComponentClass();

        if (o instanceof RendererComponent)
            return (RendererComponent) o;

        if (o instanceof RendererComponentFactory)
            return ((RendererComponentFactory) o).getComponentInstance(getParams());

        throw new IllegalStateException("Renderer component does not implement RendererComponent or RendererComponentFactory: " + o.getClass());
    }

    private void initialiseWeight(Element element) throws PluginParseException {
        String weightStr = element.attributeValue("weight");

        if (weightStr == null)
            throw new PluginParseException("Renderer component plugins must specify a weight");

        try {
            weight = Integer.parseInt(weightStr);
        } catch (NumberFormatException e) {
            throw new PluginParseException("Invalid weight, must be a number: " + weightStr);
        }
    }

    private void assertComponentOfExpectedClass() throws PluginParseException {
        Class moduleClass = getModuleClass();

        if (!RendererComponent.class.isAssignableFrom(moduleClass) && !RendererComponentFactory.class.isAssignableFrom(moduleClass))
            throw new PluginParseException("Module class must implement RendererComponent or RendererComponentFactory: " + moduleClass.getName());
    }


}
