package com.atlassian.renderer.v2.macro.basic.validator;

import org.apache.commons.lang.StringUtils;

import java.util.HashSet;
import java.util.Set;
import java.util.regex.Pattern;

/**
 * Validates that a parameter is a valid color property value.
 * <p>
 * This class was originally very strict and would only allow valid CSS2 color values. However, due to
 * CONF-10001 this restriction is currently relaxed. It would be nice to reinstate that strictness at
 * some later date.
 */
public class ColorStyleValidator implements ParameterValidator {
    public static final Pattern NAMED_COLOR_PATTERN = Pattern.compile("\\w+");
    public static final Pattern HEX_SHORT_COLOR_PATTERN = Pattern.compile("^#([\\da-fA-F]){3}$");
    public static final Pattern HEX_LONG_COLOR_MATCH = Pattern.compile("^#([\\da-fA-F]){6}$");
    public static final Pattern RGB_COLOR_PATTERN = Pattern.compile("^#rgb\\((\\d+),\\s*(\\d+),\\s*(\\d+)\\)$");
    public static final Pattern PERC_COLOR_PATTERN = Pattern.compile("^#rgb\\((\\d+)%,\\s*(\\d+)%,\\s*(\\d+)%\\)$");

    private static final ColorStyleValidator INSTANCE = new ColorStyleValidator();

    private ColorStyleValidator() {
    }

    public static ColorStyleValidator getInstance() {
        return INSTANCE;
    }

    public void assertValid(String propertyValue) throws MacroParameterValidationException {
        if (StringUtils.isBlank(propertyValue)) {
            return;
        }

        if (NAMED_COLOR_PATTERN.matcher(propertyValue).matches()
                || HEX_SHORT_COLOR_PATTERN.matcher(propertyValue).matches()
                || HEX_LONG_COLOR_MATCH.matcher(propertyValue).matches()
                || RGB_COLOR_PATTERN.matcher(propertyValue).matches()
                || PERC_COLOR_PATTERN.matcher(propertyValue).matches()) {
            return;
        }

        throw new MacroParameterValidationException("Color value is invalid");
    }

    private static Set createColorValues() {
        Set strings = new HashSet();
        strings.add("aqua");
        strings.add("black");
        strings.add("blue");
        strings.add("fuchsia");
        strings.add("gray");
        strings.add("green");
        strings.add("lime");
        strings.add("maroon");
        strings.add("navy");
        strings.add("olive");
        strings.add("orange");
        strings.add("purple");
        strings.add("red");
        strings.add("silver");
        strings.add("teal;");
        strings.add("white");
        strings.add("yellow");
        return strings;
    }
}
