/*
 * Created by IntelliJ IDEA.
 * User: Mike
 * Date: Oct 28, 2004
 * Time: 1:34:28 PM
 */
package com.atlassian.renderer.v2.components;

import com.atlassian.renderer.RenderContext;
import com.atlassian.renderer.links.Link;
import com.atlassian.renderer.links.LinkResolver;
import com.atlassian.renderer.links.UnresolvedLink;
import com.atlassian.renderer.v2.RenderMode;
import com.atlassian.renderer.v2.components.link.LinkDecorator;

import java.util.Map;
import java.util.NavigableMap;
import java.util.TreeMap;
import java.util.regex.Matcher;

public class LinkRendererComponent implements RendererComponent {
    private LinkResolver linkResolver;
    public static final char START_LINK_CHAR = '[';
    private static final char ESCAPE_CHAR = '\\';
    private static final char END_LINK_CHAR = ']';
    private static final char NEW_LINE_CHAR = '\n';

    public LinkRendererComponent(LinkResolver linkResolver) {
        this.linkResolver = linkResolver;
    }

    public boolean shouldRender(RenderMode renderMode) {
        return renderMode.renderLinks();
    }

    public String render(String wiki, RenderContext context) {
        if (wiki == null || wiki.length() < 3) {
            return wiki;
        }

        // Pre-scan
        final InsideLinkPatternCache cache = new InsideLinkPatternCache();
        int start = findLinkStart(wiki, 0, cache);
        while (start != -1) {
            int end = findLinkEnd(wiki, start);
            if (end == 0) {
                break;
            }
            if (end < 0) {
                start = findLinkStart(wiki, -end, cache);
                continue;
            }
            if (containsBracketsAndIsInvalidLink(context, wiki, start, end)) {
                start = findLinkStart(wiki, start + 1, cache);
                continue;
            }

            // Found the first link; fall through to heavy version
            return renderHeavy(wiki, context, start, end, cache);
        }
        return wiki;
    }


    private String renderHeavy(String wiki, RenderContext context, int start, int end, final InsideLinkPatternCache cache) {
        final int len = wiki.length();
        final StringBuilder sb = new StringBuilder(len / 2 * 3 + 1);
        sb.append(wiki, 0, start);
        appendLink(sb, context, wiki.substring(start + 1, end));

        int mark = end + 1;
        start = findLinkStart(wiki, mark, cache);
        while (start != -1) {
            end = findLinkEnd(wiki, start);
            if (end == 0) {
                break;
            }
            if (end < 0) {
                start = findLinkStart(wiki, -end, cache);
                continue;
            }
            if (containsBracketsAndIsInvalidLink(context, wiki, start, end)) {
                start = findLinkStart(wiki, start + 1, cache);
                continue;
            }

            sb.append(wiki, mark, start);
            appendLink(sb, context, wiki.substring(start + 1, end));
            mark = end + 1;
            start = findLinkStart(wiki, mark, cache);
        }
        return sb.append(wiki, mark, len).toString();
    }

    private boolean containsBracketsAndIsInvalidLink(RenderContext context, String wiki, int start, int end) {
        final String linkText = wiki.substring(start + 1, end);
        if (linkText.contains("[") && linkText.contains("]")) {
            Link link = linkResolver.createLink(context, linkText);
            return link instanceof UnresolvedLink;
        }
        return false;
    }

    private void appendLink(StringBuilder stringBuffer, RenderContext context, String linkText) {
        Link link = linkResolver.createLink(context, linkText);
        stringBuffer.append(context.getRenderedContentStore().addInline(new LinkDecorator(link)));
    }

    private class InsideLinkPatternCache {
        private NavigableMap<Integer, Integer> indices;

        InsideLinkPatternCache() {
            indices = null;
        }

        InsideLinkPatternCache initializeIfNecessary(final String wiki) {
            if (indices == null) {
                indices = new TreeMap<Integer, Integer>();

                final Matcher matcher = UrlRendererComponent.PATTERN_WITH_EMBED.matcher(wiki);
                while (matcher.find()) {
                    indices.put(matcher.start(), matcher.end());
                }
            }
            return this;
        }

        boolean isIndexInsidePattern(int index) {
            if (indices.isEmpty()) {
                return false;
            }

            Map.Entry<Integer, Integer> entry = indices.floorEntry(index - 1);
            return (entry != null) && entry.getValue() > index;
        }
    }

    /**
     * Checks if the index char of wiki is contained by some regex pattern. That way you can find
     * if arbitrary index is for example surrounded by ! bangs, or whether it is part of url.
     *
     * @param wiki  the text to perform the check against
     * @param index the index of the char we're testing
     * @return true if index is inside of one of the pattern matches, false otherwise
     */
    private static boolean isInsideAnyLinkPattern(final String wiki, int index, final InsideLinkPatternCache cache) {
        return cache.initializeIfNecessary(wiki).isIndexInsidePattern(index);
    }

    /**
     * Looks for the start of a well-formed link.  The start link char is ignored
     * if the start link char is
     * <ul>
     * <li>escaped, or</li>
     * <li>at the end of the string, or</li>
     * <li>immediately followed by the end link char (empty link), or</li>
     * <li>immediately followed by whitespace</li>
     * </ul>
     *
     * @param wiki       the text to scan for links
     * @param startIndex the index at which to begin scanning
     * @return the index of the {@link #START_LINK_CHAR}, or <code>-1</code> if
     * the end of the string is reached without finding one
     */
    private static int findLinkStart(String wiki, int startIndex, final InsideLinkPatternCache cache) {
        final int len = wiki.length();

        int index = wiki.indexOf(START_LINK_CHAR, startIndex);
        for (; index != -1; index = wiki.indexOf(START_LINK_CHAR, index + 1)) {
            // Ignore \[
            if (index > 0 && wiki.charAt(index - 1) == ESCAPE_CHAR) {
                continue;
            }

            // Ignore [<end>
            final int nextIndex = index + 1;
            if (nextIndex == len) {
                return -1;
            }

            // Ignore []
            final char nextChar = wiki.charAt(nextIndex);
            if (nextChar == END_LINK_CHAR || Character.isWhitespace(nextChar)) {
                continue;
            }

            // Ignore links that are between 'embedded object' bangs ('!') or inside casual url (see RNDR-144)
            if (isInsideAnyLinkPattern(wiki, index, cache)) {
                continue;
            }

            break;
        }
        return index;
    }

    /**
     * Looks for the end of a well-formed link.  The return value indicates
     * what was found as follows:
     * <table>
     * <tr><th>Value</th><th>Meaning</th></tr>
     * <tr><td>&gt; 0</td><td>Found the end of the link, and the return value is
     * the index of the {@link #END_LINK_CHAR}.</td></tr>
     * <tr><td>0</td><td>Reached the end of the string</td></tr>
     * <tr><td>&lt; 0</td><td>The link is not well-formed, and the return value
     * is the negative of the index at which the scan should resume.</td></tr>
     * </table>
     *
     * @param wiki       the text to be rendered
     * @param startIndex the index of the {@link #START_LINK_CHAR} to be matched
     * @return see description
     */
    private static int findLinkEnd(String wiki, final int startIndex) {
        final int len = wiki.length();
        char prev = 0;

        int startLinkCount = 0;
        int lastStartLinkIdx = 0;
        int endLinkCount = 0;

        for (int index = startIndex + 1; index < len; ++index) {
            final char c = wiki.charAt(index);

            switch (c) {
                case END_LINK_CHAR: {
                    if (prev != ESCAPE_CHAR) {
                        ++endLinkCount;
                        //finish on well formatted closing bracket
                        if (endLinkCount == startLinkCount + 1) {
                            return index;
                        } else {
                            continue;
                        }
                    }
                    break;
                }

                case START_LINK_CHAR: {
                    if (prev != ESCAPE_CHAR) {
                        ++startLinkCount;
                        //allow only one additional opening bracked inside link (ipv6 syntax)
                        //if second opening bracket is found then start over from last opened bracket
                        if (startLinkCount > 1) {
                            return -lastStartLinkIdx;
                        } else {
                            lastStartLinkIdx = index;
                            continue;
                        }
                    }
                    break;
                }

                case NEW_LINE_CHAR: {
                    //if end of string is reached and we have unclosed brackets then start over from unclosed opening bracket
                    if (startLinkCount > 0) {
                        return -lastStartLinkIdx;
                    }
                    // \n -> restart scan from next char (whether escaped or not)
                    return -(index + 1);
                }
            }

            prev = c;
        }

        //if end of string is reached and we have unclosed brackets then start over from unclosed opening bracket
        if (startLinkCount > 0) {
            return -lastStartLinkIdx;
        }
        // Hit the end of the string
        return 0;
    }
}