package com.atlassian.renderer.v2.components;

import com.atlassian.renderer.RenderContext;
import com.atlassian.renderer.v2.RenderMode;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * Escapes certain symbols prefixed with backslash (\) into html entities.
 * Most symbols in wiki markup <b>cannot</b> be escaped with wiki markup, so that the
 * backslash character can be interpreted in a literal most of the time.  This makes
 * it easier to do things such as pasting windows file paths into wiki markup.
 */
public class BackslashEscapeRendererComponent extends AbstractRegexRendererComponent {
    // These two constants should be updated together.
    private static final Pattern ESCAPING_PATTERN = Pattern.compile("(^|(?<!\\\\))\\\\([\\-\\#\\*\\_\\+\\?\\^\\~\\|\\%\\{\\}\\[\\]\\(\\)\\!\\@])");

    /**
     * This method will escape with a backslash the characters that are rendered by the {@link #render} method.
     * This is probably not the best method to use to escape anything so if possible do not call it.
     */
    public static String escapeWiki(String str) {
        int i = findNextSpecialChar(str, 0);
        if (i == -1) {
            return str;
        }

        final int len = str.length();
        final StringBuilder sb = new StringBuilder(len + 10);
        int mark = 0;

        do {
            sb.append(str, mark, i).append('\\');
            mark = i;
            i = findNextSpecialChar(str, i + 1);
        }
        while (i != -1);
        return sb.append(str, mark, len).toString();
    }

    private static int findNextSpecialChar(String str, int index) {
        final int len = str.length();
        while (index < len) {
            if (isSpecial(str.charAt(index))) {
                return index;
            }
            ++index;
        }
        return -1;
    }

    private static boolean isSpecial(final char c) {
        switch (c) {
            case '!':
            case '[':
            case ']':
            case '^':
            case '~':
            case '+':
            case '?':
            case '%':
            case '{':
            case '}':
            case '(':
            case ')':
            case '*':
            case '_':
            case '-':
            case '|':
            case '@':
                return true;
            default:
                return false;
        }
    }

    public boolean shouldRender(RenderMode renderMode) {
        return renderMode.backslashEscape();
    }

    public String render(String wiki, RenderContext context) {
        return (wiki.indexOf('\\') == -1) ? wiki : regexRender(wiki, context, ESCAPING_PATTERN);
    }

    public void appendSubstitution(StringBuffer buffer, RenderContext context, Matcher matcher) {
        final char c = matcher.group(2).charAt(0);

        // Because the escape filter runs after the macro and link filters, those filters have to take care of
        // their own escaping, but we
        switch (c) {
            case '[':
            case ']':
                buffer.append(c);
                return;

            case '{':
            case '}':
                if (context.isRenderingForWysiwyg()) {
                    buffer.append(context.getRenderedContentStore().addInline("\\" + c));
                    return;
                }
                buffer.append(c);
                return;

            default:
                buffer.append("&#").append(((int) c)).append(';');
        }
    }
}
