package com.atlassian.renderer.v2;

import com.atlassian.renderer.RenderContext;
import com.atlassian.renderer.RendererConfiguration;
import com.atlassian.renderer.WikiStyleRenderer;
import com.atlassian.renderer.embedded.EmbeddedResourceRenderer;
import com.atlassian.renderer.links.LinkRenderer;
import com.atlassian.renderer.util.UnicodeEscapingTool;
import com.opensymphony.util.TextUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * The facade class for the whole rendering system. This is what you call if you want to render something from
 * anywhere OUTSIDE the rendering subsystem. The facade is responsible for setting up the environment before
 * the renderer is invoked, if there is anything missing.
 * <p/>
 * Inside the rendering subsystem, you are more likely to call the V2Renderer directly (or the SubRenderer)
 */
public class V2RendererFacade implements WikiStyleRenderer {
    private static final Logger log = LoggerFactory.getLogger(V2RendererFacade.class);
    private RendererConfiguration rendererConfiguration;
    private LinkRenderer defaultLinkRenderer;
    private EmbeddedResourceRenderer defaultEmbeddedRenderer;
    private TokenEscapingV2Renderer renderer;
    private boolean escapeInternalTokens = true;
    private UnicodeEscapingTool defaultUnicodeEscapingTool;

    public V2RendererFacade() {
    }

    public V2RendererFacade(RendererConfiguration rendererConfiguration,
                            LinkRenderer defaultLinkRenderer,
                            EmbeddedResourceRenderer defaultEmbeddedRenderer,
                            Renderer renderer,
                            UnicodeEscapingTool defaultUnicodeEscapingTool,
                            boolean escapeInternalTokens) {
        this.rendererConfiguration = rendererConfiguration;
        this.defaultLinkRenderer = defaultLinkRenderer;
        this.defaultEmbeddedRenderer = defaultEmbeddedRenderer;
        this.defaultUnicodeEscapingTool = defaultUnicodeEscapingTool;
        this.renderer = new TokenEscapingV2Renderer(renderer, escapeInternalTokens);
        this.escapeInternalTokens = escapeInternalTokens;
        logSanitizer(escapeInternalTokens);
    }

    private void logSanitizer(boolean escapeInternalTokens) {
        if (escapeInternalTokens) {
            log.debug("Enabled wiki markup sanitizer");
        } else {
            log.info("Disabled wiki markup sanitizer");
        }
    }

    public void setRendererConfiguration(RendererConfiguration rendererConfiguration) {
        this.rendererConfiguration = rendererConfiguration;
    }


    public void setRenderer(Renderer renderer) {
        this.renderer = new TokenEscapingV2Renderer(renderer, escapeInternalTokens);
    }

    public void setEscapeInternalTokens(boolean escapeInternalTokens) {
        this.escapeInternalTokens = escapeInternalTokens;
        if (this.renderer != null) {
            this.renderer.setEscapeInternalTokens(escapeInternalTokens);
        }
        logSanitizer(escapeInternalTokens);
    }

    public boolean isEscapeInternalTokensEnabled() {
        return escapeInternalTokens;
    }

    public String convertWikiToXHtml(RenderContext context, String wiki) {
        if (!TextUtils.stringSet(wiki)) {
            return "";
        }

        if (context.getRenderMode() != null && context.getRenderMode().renderNothing()) {
            return wiki;
        }

        initializeContext(context);

        return renderer.render(wiki, context);
    }

    public String convertWikiToText(RenderContext context, String wiki) {
        if (!TextUtils.stringSet(wiki)) {
            return "";
        }

        if (context.getRenderMode().renderNothing()) {
            return wiki;
        }

        initializeContext(context);

        return renderer.renderAsText(wiki, context);
    }

    public void setDefaultLinkRenderer(LinkRenderer linkRenderer) {
        this.defaultLinkRenderer = linkRenderer;
    }

    public void setDefaultEmbeddedRenderer(EmbeddedResourceRenderer embeddedRenderer) {
        this.defaultEmbeddedRenderer = embeddedRenderer;
    }

    public void setDefaultUnicodeEscapingTool(UnicodeEscapingTool defaultUnicodeEscapingTool) {
        this.defaultUnicodeEscapingTool = defaultUnicodeEscapingTool;
    }

    private void initializeContext(RenderContext context) {
        if (context.getSiteRoot() == null) {
            context.setSiteRoot(rendererConfiguration.getWebAppContextPath());
        }

        if (context.getImagePath() == null) {
            context.setImagePath(context.getSiteRoot() + "/images");
        }

        if (context.getLinkRenderer() == null) {
            context.setLinkRenderer(defaultLinkRenderer);
        }

        if (context.getEmbeddedResourceRenderer() == null) {
            context.setEmbeddedResourceRenderer(defaultEmbeddedRenderer);
        }

        if (context.getCharacterEncoding() == null) {
            context.setCharacterEncoding(rendererConfiguration.getCharacterEncoding());
        }

        if(context.getSetUnicodeEscapingTool() == null) { // because we are trying to be safe
            context.setUnicodeEscapingTool(defaultUnicodeEscapingTool);
        }

    }

}
