package com.atlassian.renderer.v2;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * Helper class for performing repeated pattern replacement.
 * Why you can't do this with just the JDK regex classes, I don't know.
 */
public class Replacer {
    /**
     * When we notice that we need to perform a replacement, we don't really
     * have any way of knowing for certain exactly how much space the buffer
     * needs to have.  This is a hardcoded bit-of-extra-space that we grant
     * the buffer up front so that one or two small increases won't force an
     * additional copy of the data.  The value used here is just a wild stab
     * in the dark.
     */
    private static final int BUFFER_PADDING = 64;

    /**
     * An empty String array for convenient use as the <code>necessaryConstantParts</code>
     * argument to {@link #Replacer(Pattern, String, String...)}.
     */
    private static final String[] EMPTY_ARRAY = {};

    private final Pattern pattern;
    private final String replacement;
    private final String[] necessaryConstantParts;
    private final boolean replacementHasMetachars;

    /**
     * Convenience constructor which will not require any constant parts
     * (the replacement is always attempted).
     *
     * @param pattern     as for {@link #Replacer(Pattern, String, String...)}}
     * @param replacement as for {@link #Replacer(Pattern, String, String...)}
     */
    public Replacer(Pattern pattern, String replacement) {
        this(pattern, replacement, EMPTY_ARRAY);
    }

    /**
     * Replaces a pattern with a replacement, optionally provided with an array of necessary
     * constant parts.  The {@link #replaceAll(String)} method will not evaluate the regular
     * expression patterns unless the target string contains all of the necessary constant
     * parts.  This is a performance enhancement, as substring searches are cheaper than
     * regular expression evaluation.
     *
     * @param pattern                the regular expression pattern to match
     * @param replacement            the replacement expression
     * @param necessaryConstantParts substrings that are checked by {@link #replaceAll(String)} before
     *                               evaluating the regular expressions.
     */
    public Replacer(Pattern pattern, String replacement, String... necessaryConstantParts) {
        this.pattern = pattern;
        this.replacement = replacement;
        this.necessaryConstantParts = necessaryConstantParts;
        this.replacementHasMetachars = replacement.indexOf('\\') != -1 || replacement.indexOf('$') != -1;
    }

    /**
     * Checks that <code>str</code> contains all of the required parts, and if
     * so, calls {@link #replaceAllSkippingConstantsCheck(String)} to perform the
     * actual replacement.
     *
     * @param str the string to perform replacements upon
     * @return <code>str</code> if the string did not satisfy the
     * <code>necessaryConstantParts</code> or otherwise did not
     * need to be modified; otherwise, the modified string after
     * replacements have been performed
     * @throws IndexOutOfBoundsException as for {@link #replaceAllSkippingConstantsCheck(String)}
     * @throws IllegalArgumentException  as for {@link #replaceAllSkippingConstantsCheck(String)}
     */
    public String replaceAll(String str) {
        // Note: Deliberately avoiding the enhanced for loop and String.contains(String)
        // because they benchmark at nearly 10% slower and this method is *very* heavily
        // used.

        // noinspection ForLoopReplaceableByForEach
        for (int i = 0; i < necessaryConstantParts.length; ++i) {
            // noinspection IndexOfReplaceableByContains
            if (str.indexOf(necessaryConstantParts[i]) == -1) {
                return str;
            }
        }
        return replaceAllSkippingConstantsCheck(str);
    }

    /**
     * This method does exactly the same thing as {@link #replaceAll(String)}, except that
     * it bypasses the check for the <code>necessaryConstantParts</code> that were provided
     * in the {@link #Replacer(Pattern, String, String...) constructor}.  The name of this
     * method is unclear and might be mistaken as a <code>replaceFirst</code> operation, so
     * it has been renamed to {@link #replaceAllSkippingConstantsCheck(String)} for clarity.
     *
     * @param str the string to perform replacement on
     * @return the string after replacement
     * @throws IndexOutOfBoundsException as for {@link #replaceAllSkippingConstantsCheck(String)}
     * @throws IllegalArgumentException  as for {@link #replaceAllSkippingConstantsCheck(String)}
     * @deprecated since 7.1 - use {@link #replaceAllSkippingConstantsCheck(String)} instead.
     */
    @Deprecated
    public String replace(String str) {
        return replaceAllSkippingConstantsCheck(str);
    }

    /**
     * This method does exactly the same thing as {@link #replaceAll(String)}, except that
     * it bypasses the check for the <code>necessaryConstantParts</code> that were provided
     * in the {@link #Replacer(Pattern, String, String...) constructor}.
     *
     * @param str the string to perform replacement on
     * @return the string after replacement
     * @throws IndexOutOfBoundsException if the replacement string ends with
     *                                   an unescaped backslash (<code>\</code>) or dollar-sign (<code>$</code>),
     *                                   or if a group reference is present that has no matching group,
     *                                   such as <code>$2</code> when the regular expression pattern only had
     *                                   one capturing group.
     * @throws IllegalArgumentException  if the replacement string includes a
     *                                   an unescaped dollar-sign (<code>$</code>) that is immediately
     *                                   followed by a non-digit, such as (<code>$x</code>).
     */
    public String replaceAllSkippingConstantsCheck(String str) {
        final Matcher matcher = pattern.matcher(str);
        if (!matcher.find()) {
            return str;
        }

        if (replacementHasMetachars) {
            return replaceAllHeavy(str, matcher);
        }

        // Simple version that does literal replacements
        final StringBuilder sb = new StringBuilder(str.length() + BUFFER_PADDING)
                .append(str, 0, matcher.start())
                .append(replacement);
        int mark = matcher.end();
        while (matcher.find()) {
            sb.append(str, mark, matcher.start()).append(replacement);
            mark = matcher.end();
        }
        return sb.append(str, mark, str.length()).toString();
    }

    private String replaceAllHeavy(String str, Matcher matcher) {
        final StringBuilder sb = new StringBuilder(str.length() + BUFFER_PADDING)
                .append(str, 0, matcher.start());
        appendReplacementPattern(sb, matcher);
        int mark = matcher.end();

        while (matcher.find()) {
            sb.append(str, mark, matcher.start());
            appendReplacementPattern(sb, matcher);
            mark = matcher.end();
        }
        return sb.append(str, mark, str.length()).toString();
    }

    /**
     * Appends the replacement pattern to the string builder using the
     * current state of the matcher to provide reference substitutions.
     * <p/>
     * <strong>WARNING:</strong> This is <strong>not</strong> equivalent to
     * {@link Matcher#appendReplacement(StringBuffer, String)}!  That method
     * keeps track of the previous <code>matcher.end()</code> value internally
     * so that it can append the text between the previous match and the
     * current one.  This method does <strong>not</strong> do that.  The
     * caller is responsible for making the appropriate call to
     * <code>sb.append(str, mark, matcher.start())</code> before calling
     * this method.
     *
     * @param sb      the string builder to receive the interpreted replacement
     *                pattern
     * @param matcher the matcher containing state from a call to
     *                {@link Matcher#find()} that will supply the values
     *                for group references.
     * @throws IndexOutOfBoundsException if the replacement string ends with
     *                                   an unescaped backslash (<code>\</code>) or dollar-sign (<code>$</code>),
     *                                   or if a group reference is present that has no matching group,
     *                                   such as <code>$2</code> when the regular expression pattern only had
     *                                   one capturing group.  This does not occur unless at least one match
     *                                   is found.
     * @throws IllegalArgumentException  if the replacement string includes a
     *                                   an unescaped dollar-sign (<code>$</code>) that is immediately
     *                                   followed by a non-digit, such as (<code>$x</code>).  This does not
     *                                   occur unless at least one match is found.
     */
    private void appendReplacementPattern(StringBuilder sb, Matcher matcher) {
        final int len = replacement.length();
        final int groupCount = matcher.groupCount();

        int cursor = 0;
        while (cursor < len) {
            final char c = replacement.charAt(cursor++);
            if (c == '\\') {
                // Note: IndexOutOfBoundsException if '\' is at the end of the string
                sb.append(replacement.charAt(cursor++));
                continue;
            }

            if (c != '$') {
                sb.append(c);
                continue;
            }

            // Note: IndexOutOfBoundsException if '$' is at the end of the string
            int group = replacement.charAt(cursor) - '0';
            if (group < 0 || group > 9) {
                throw new IllegalArgumentException("Bad group reference: $" + replacement.charAt(cursor));
            }

            while (++cursor < len) {
                int digit = replacement.charAt(cursor) - '0';
                if (digit < 0 || digit > 9) {
                    break;
                }

                int newGroup = group * 10 + digit;
                if (groupCount < newGroup) {
                    break;
                }

                group = newGroup;
            }

            // Note: IndexOutOfBoundsException if the group doesn't exist
            final String groupText = matcher.group(group);
            if (groupText != null) {
                sb.append(groupText);
            }
        }
    }
}

