package com.atlassian.renderer.wysiwyg.converter;

import com.atlassian.renderer.v2.RenderUtils;
import com.atlassian.renderer.v2.macro.Macro;
import com.atlassian.renderer.wysiwyg.NodeContext;
import com.atlassian.renderer.wysiwyg.WysiwygMacroHelper;
import org.w3c.dom.Node;

/**
 * Converts a span node to wiki markup.  Spans may contain macros that need to be handled by the WysiwygMacroHelper,
 * wiki source divs that are converted as raw text or errors that are swallowed.  All other spans have their children
 * converted normally.
 */
public class SpanConverter implements Converter {
    static final SpanConverter INSTANCE = new SpanConverter();

    public boolean canConvert(NodeContext nodeContext) {
        return nodeContext.getNode().getNodeName().equalsIgnoreCase("span");
    }

    public String convertNode(NodeContext nodeContext, DefaultWysiwygConverter wysiwygConverter) {
        Node node = nodeContext.getNode();
        String className = nodeContext.getAttribute("class");
        if (className != null) {
            String macroName = WysiwygMacroHelper.getMacroName(node);
            if (macroName != null) {
                Macro macro = wysiwygConverter.getMacroManager().getEnabledMacro(macroName);
                // text children of a macro node definitely need to be converted. 
                NodeContext contextWithConvertText = new NodeContext.Builder(nodeContext).ignoreText(false).build();
                return WysiwygMacroHelper.convertMacroFromNode(contextWithConvertText, wysiwygConverter, macro);
            }
            if (className.indexOf(RenderUtils.WIKI_SRC_CLASS) != -1) {
                return DefaultWysiwygConverter.getRawChildText(node, false); // this span contains wiki text which was not understood, which should be preserved
            }
            if (className.indexOf("error") != -1) {
                return ""; // this span contains error text which must be discarded
            }
        }
        // this is a span which we are ignoring
        return wysiwygConverter.convertChildren(nodeContext);
    }
}
