package com.atlassian.renderer.v2.components.block;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.StringReader;
import java.util.LinkedList;

/**
 * Iterates over a String line by line.  Acts somewhat like an iterator, but with extra methods. It provides a
 * "{@link #peek()}" function so that following lines can be checked without modifying the state of the walker,
 * and a "{@link #pushBack(String)}" function so that the state can be rolled back (or even falsified) when
 * required.
 * <p>
 * This class is not thread-safe.
 */
public class LineWalker {
    private BufferedReader reader;
    /**
     * Queue of the next blocks to be returned.
     */
    private LinkedList<String> queue = new LinkedList<String>();

    public LineWalker(String text) {
        reader = new BufferedReader(new StringReader(text));
    }

    /**
     * @return true if there are more lines to be walked
     */
    public boolean hasNext() {
        if (queue.isEmpty())
            readNextLineToQueue();
        return !queue.isEmpty();
    }

    /**
     * Returns the next line to be walked <b>without actually walking it</b>.  Throws an
     * exception if there are no lines, so call {@link #hasNext} first.
     *
     * @return the next line to be walked.
     * @throws IllegalStateException if there are no more lines to be walked.
     */
    public String peek() {
        if (!hasNext())
            throw new IllegalStateException("No more lines");

        return queue.getFirst();
    }

    /**
     * Walks the next line, returning the line and advancing the state of the walker past that line.  Throws an
     * exception if there are no lines, so call {@link #hasNext} first.
     *
     * @return the next line.
     * @throws IllegalStateException if there are no more lines to be walked.
     */
    public String next() {
        if (!hasNext())
            throw new IllegalStateException("No more lines");
        return queue.removeFirst();
    }

    /**
     * Read the next line from {@link #reader}.  Can fail with runtime exception but shouldn't.  Will return null
     * if reader is finished.
     *
     * @throws RuntimeException if an IOException occurs while reading from {@link #reader}
     */
    private void readNextLineToQueue() {
        if (reader == null)
            return;
        try {
            final String result = reader.readLine();
            if (result != null) {
                queue.add(result);
            } else {
                reader.close();
                reader = null;
            }
        } catch (IOException e) {
            throw new RuntimeException("IO Exception reading from string: " + e.getMessage(), e);
        }
    }

    public void pushBack(String line) {
        queue.addFirst(line);
    }
}
