package com.atlassian.renderer.v2.components;

import com.atlassian.renderer.RenderContext;
import com.atlassian.renderer.v2.RenderMode;
import com.atlassian.renderer.v2.RenderUtils;
import com.atlassian.renderer.v2.SubRenderer;
import com.atlassian.renderer.v2.WikiMarkupParser;
import com.atlassian.renderer.v2.macro.Macro;
import com.atlassian.renderer.v2.macro.MacroException;
import com.atlassian.renderer.v2.macro.MacroManager;
import com.atlassian.renderer.wysiwyg.WysiwygMacroHelper;
import com.atlassian.util.profiling.UtilTimerStack;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Date;
import java.util.HashMap;
import java.util.Map;

public class MacroRendererComponent extends AbstractRendererComponent {
    private static final Logger log = LoggerFactory.getLogger(MacroRendererComponent.class);

    private final MacroManager macroManager;
    private final SubRenderer subRenderer;
    private final WysiwygMacroHelper wysiwygMacroHelper;

    private static final Logger timeLogger = LoggerFactory.getLogger("macroLogger");

    /**
     * By default if not provided by ${@link RenderContext#getParam(Object)}, this parameter will be true, otherwise it is
     * converted to a boolean to determine whether should consider ${...} as the variable format, and therefore skip in
     * regards to our macro processing.
     */
    public static final String CONSIDER_VARIABLE_FORMAT_PARAMETER = "com.atlassian.renderer.v2.components.macro.render.consider.variable.format";

    public MacroRendererComponent(MacroManager macroManager, SubRenderer subRenderer) {
        this.macroManager = macroManager;
        this.subRenderer = subRenderer;
        this.wysiwygMacroHelper = new WysiwygMacroHelper(this);
    }

    public boolean shouldRender(RenderMode renderMode) {
        return renderMode.renderMacros();
    }

    public String render(String wiki, RenderContext context) {
        WikiMarkupParser parser = new WikiMarkupParser(macroManager, new WikiContentRendererHandler(this, context));
        return parser.parse(wiki, considerVariableFormat(context));
    }

    public void makeMacro(StringBuffer buffer, MacroTag startTag, String body, RenderContext context) {
        Date now = new Date();
        Macro macro = getMacroByName(startTag.command);
        Map<String, Object> params = makeParams(startTag.argString);

        if (RssThreadLocal.inRss.get() && RssThreadLocal.blackListedMacros.get().contains(startTag.command)) {
            buffer.append("Macro " + startTag.command + " cannot be used within RSS requests");
            return;
        }
        if (context.isRenderingForWysiwyg()) {
            wysiwygMacroHelper.renderMacro(startTag, macro, body, params, context, buffer);
        } else {
            if (macro != null) {
                processMacro(startTag.command, macro, body, params, context, buffer);
            } else {
                handleUnknownMacroTag(buffer, startTag, body, context);
            }
        }
        Date after = new Date();
        if (RssThreadLocal.inRss.get() && timeLogger.isDebugEnabled()) {
            long entityId = -1;
            timeLogger.debug(startTag + "=" + (after.getTime() - now.getTime()) + ":" + entityId);
        }

    }

    private void handleUnknownMacroTag(StringBuffer buffer, MacroTag startTag, String body, RenderContext context) {
        if (!context.getRenderMode().renderMacroErrorMessages()) {
            HtmlEscapeRendererComponent htmlEscapeRendererComponent = new HtmlEscapeRendererComponent();
            StringBuffer errorBuffer = new StringBuffer();
            // there must be a more efficient way to do this escaping, it seems there are not simple html
            // escape utilities available to the renderer other than this component
            errorBuffer.append(htmlEscapeRendererComponent.render(startTag.originalText, context));
            // append the body and closing tags if there is one
            if (StringUtils.isNotBlank(body)) {
                errorBuffer.append(subRenderer.render(body, context, context.getRenderMode().and(RenderMode.suppress(RenderMode.F_FIRST_PARA | RenderMode.F_PARAGRAPHS))));
                errorBuffer.append("{").append(htmlEscapeRendererComponent.render(startTag.command, context)).append("}");
            }
            buffer.append(context.addRenderedContent(errorBuffer.toString()));
        } else {
            buffer.append(makeMacroError(context, "Unknown macro: {" + startTag.command + "}", body));
        }
    }

    private Macro getMacroByName(String name) {
        if (name == null) {
            return null;
        }

        return macroManager.getEnabledMacro(name.toLowerCase());
    }

    private Map<String, Object> makeParams(String paramString) {
        Map<String, Object> params = new HashMap<String, Object>();
        params.put(Macro.RAW_PARAMS_KEY, paramString == null ? "" : paramString);

        if (StringUtils.isEmpty(paramString)) {
            return params;
        }

        String[] paramStrs = paramString.split("\\|");

        for (int i = 0; i < paramStrs.length; i++) {
            String paramStr = paramStrs[i];
            int idx;

            if ((idx = paramStr.indexOf("=")) != -1) {
                if (idx == paramStr.length() - 1) {
                    params.put(paramStr.substring(0, idx).trim(), "");
                } else {
                    params.put(paramStr.substring(0, idx).trim(), paramStr.substring(idx + 1).trim());
                }
            } else {
                params.put(String.valueOf(i), paramStr);
            }
        }

        return params;
    }

    public void processMacro(String command, Macro macro, String body, Map<String, Object> params, RenderContext context, StringBuffer buffer) {
        String renderedBody = body;
        try {
            if (StringUtils.isNotEmpty(body) && macro.getBodyRenderMode() != null && !macro.getBodyRenderMode().renderNothing()) {
                // Get rid of the leading \n if this is for wysiwig, it gets confused as a blank line and the converter
                // will try and preserve it as a blank line put there by the user.
                RenderMode macroMode = macro.getBodyRenderMode();
                if (context.isRenderingForWysiwyg() && macroMode.renderParagraphs()) {
                    renderedBody = RenderUtils.trimInitialNewline(renderedBody);
                }
                renderedBody = subRenderer.render(renderedBody, context, macroMode);
            }

            String macroResult = executeMacro(command, macro, params, renderedBody, context);

            if (macro.getBodyRenderMode() == null) {
                /* We need to render the macros only here as the other render components will be run after this one.
                   This is to ensure macros with macros inside the body also get rendered correctly. (RNDR-4) */
                buffer.append(subRenderer.render(macroResult, context, RenderMode.MACROS_ONLY));
            } else {
                buffer.append(context.addRenderedContent(macroResult, macro.getTokenType(params, body, context)));
            }

        } catch (MacroException e) {
            log.info("Error formatting macro: " + command + ": " + e, e);
            buffer.append(makeMacroError(context, command + ": " + e.getMessage(), body));
        } catch (Throwable t) {
            log.error("Unexpected error formatting macro: " + command, t);
            buffer.append(makeMacroError(context, "Error formatting macro: " + command + ": " + t.toString(), body));
        }
    }

    protected String executeMacro(String command, Macro macro, Map<String, Object> params, String renderedBody, RenderContext context) throws MacroException {
        String profilingName = "Rendering macro: {" + command + "}";
        long startTime = System.currentTimeMillis();

        try {
            UtilTimerStack.push(profilingName);
            return macro.execute(params, renderedBody, context);
        } finally {
            log.debug("Rendering macro \\{{}} took {} ms with parameters: {}", new Object[]{
                    command, (System.currentTimeMillis() - startTime), params});
            UtilTimerStack.pop(profilingName);
        }
    }

    private String makeMacroError(RenderContext context, String errorMessage, String body) {
        return context.addRenderedContent(RenderUtils.blockError(errorMessage, renderErrorBody(body, context)));
    }

    private String renderErrorBody(String body, RenderContext context) {
        return context.addRenderedContent(subRenderer.render(body, context, null));
    }

    public SubRenderer getSubRenderer() {
        return subRenderer;
    }

    /* VisibleForTesting */
    static boolean considerVariableFormat(final RenderContext renderContext) {
        final Object param = renderContext.getParam(CONSIDER_VARIABLE_FORMAT_PARAMETER);
        if (param == null) {
            // default to true if it is not provided
            return true;
        } else if (param instanceof Boolean) {
            return (Boolean) param;
        } else {
            return Boolean.parseBoolean(param.toString());
        }
    }

}
