package com.atlassian.renderer.util;

import com.atlassian.renderer.wysiwyg.NodeContext;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;

/**
 * Utility methods for {@link org.dom4j.Node nodes}.
 */
public class NodeUtil {
    /**
     * Retrieve the attribute value from the node. Null is returned if the attribute is not found.
     *
     * @see NodeContext#getAttribute(String)
     */
    public static String getAttribute(Node node, String name) {
        NamedNodeMap map = node.getAttributes();
        if (map == null) {
            // this should never happen. Comment nodes have no attributes, but they should be filtered out
            // before getAttributes() is called
            return null;
        } else {
            Node n = node.getAttributes().getNamedItem(name);
            return n != null ? n.getNodeValue() : null;
        }
    }

    /**
     * Retrieve the value of an attribute for a node as a boolean.  If the value is not
     * present, or not readily convertible to a boolean return the requested defaultValue.
     *
     * @param node          from which to retrieve the value
     * @param attributeName the attribute to find the value of
     * @param defaultValue  the value to return if no boolean value is found.
     * @return the value of an attribute
     * @see NodeContext#getBooleanAttributeValue(String, boolean)
     */
    public static boolean getBooleanAttributeValue(Node node, String attributeName, boolean defaultValue) {
        String attributeValue = getAttribute(node, attributeName);
        return String.valueOf(!defaultValue).equalsIgnoreCase(attributeValue) ? !defaultValue : defaultValue;
    }

    /**
     * Returns true if the named attribute exists and contains the passed in value.
     */
    public static boolean attributeContains(Node node, String name, String value) {
        final String attr = getAttribute(node, name);
        return attr != null && attr.indexOf(value) != -1;
    }

    public static boolean isTextNode(Node node) {
        return node.getNodeType() == Node.TEXT_NODE;
    }

    public static boolean isList(Node node) {
        return node.getNodeName().toLowerCase().equals("ol") || node.getNodeName().toLowerCase().equals("ul");
    }
}
