package com.atlassian.renderer.embedded;

import com.atlassian.renderer.RenderContext;
import com.atlassian.renderer.RenderContextOutputType;
import com.atlassian.renderer.attachments.RendererAttachment;
import com.atlassian.renderer.attachments.RendererAttachmentManager;
import com.atlassian.renderer.v2.RenderUtils;
import com.atlassian.renderer.v2.components.HtmlEscaper;
import com.atlassian.renderer.v2.macro.basic.validator.ColorStyleValidator;
import com.atlassian.renderer.v2.macro.basic.validator.CssSizeValidator;
import com.atlassian.renderer.v2.macro.basic.validator.MacroParameterValidationException;
import com.atlassian.renderer.v2.macro.basic.validator.ParameterValidator;
import com.opensymphony.util.TextUtils;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.SortedMap;
import java.util.TreeMap;

public class EmbeddedImageRenderer implements EmbeddedResourceRenderer {
    private static final String BORDERCOLOR_KEY = "bordercolor";
    private static final String BORDER_KEY = "border";

    /**
     * Contain information used to handle wiki markup attributes that will be rendered as CSS style value
     *
     * @author pcurren
     */
    private static class StyleInformation {
        String cssStyleAttribute;
        ParameterValidator valueValidator;

        StyleInformation(String attr, ParameterValidator validator) {
            cssStyleAttribute = attr;
            valueValidator = validator;
        }
    }

    private static final Map/*<String,StyleInformation>*/ STYLE_ATTRIBUTES_MAPPING = createStyleAttributeMappings();

    private RendererAttachmentManager attachmentManager;

    /**
     * Create the mappings of wiki markup attributes to the CSS style attributes they will be rendered as.
     *
     * @return A Map where the keys are Strings representing the wiki markup attributes and the values are
     * a String representing the CSS style attribute they value will be rendered as.
     */
    private static Map/*<String,StyleInformation>*/ createStyleAttributeMappings() {
        Map/*<String,StyleInformation>*/ result = new HashMap();
        StyleInformation borderWidthInfo = new StyleInformation(BORDER_KEY, CssSizeValidator.getInstance());
        StyleInformation borderColorInfo = new StyleInformation("border-color", ColorStyleValidator.getInstance());

        result.put(BORDER_KEY, borderWidthInfo);
        result.put(BORDERCOLOR_KEY, borderColorInfo);
        return result;
    }

    public EmbeddedImageRenderer(RendererAttachmentManager attachmentManager) {
        this.attachmentManager = attachmentManager;
    }

    public RendererAttachmentManager getAttachmentManager() {
        return attachmentManager;
    }

    public void setAttachmentManager(RendererAttachmentManager attachmentManager) {
        this.attachmentManager = attachmentManager;
    }

    public String renderResource(EmbeddedResource resource, RenderContext context) {
        String token;

        EmbeddedImage image = (EmbeddedImage) resource;

        RendererAttachment attachment = null;
        if (!image.isExternal()) {
            try {
                attachment = getAttachment(context, resource);
            } catch (RuntimeException re) {
                return context.addRenderedContent(RenderUtils.error(re.getMessage()));
            }
        }

        Map<Object, Object> imageParams = new HashMap<Object, Object>();
        imageParams.putAll(image.getProperties());
        if (context.isRenderingForWysiwyg()) {
            imageParams.put("imagetext", resource.getOriginalLinkText());
        }
        if (image.isThumbNail()) {
            if (image.isExternal()) {
                token = context.addRenderedContent(RenderUtils.error(context, "Can only create thumbnails for attached images", originalLink(resource), false));
            } else if (!attachmentManager.systemSupportsThumbnailing()) {
                token = context.addRenderedContent(RenderUtils.error(context, "This installation can not generate thumbnails: no image support in Java runtime", originalLink(resource), false));
            } else if (attachment == null && !context.isRenderingForWysiwyg()) {
                token = context.addRenderedContent(RenderUtils.error(context, "Attachment '" + image.getFilename() + "' was not found", originalLink(resource), false));
            } else {
                token = context.addRenderedContent(generateThumbnail(imageParams, attachment, context, image, resource));
            }
        } else {
            String imageUrl = "";

            if (image.isExternal()) {
                imageUrl = image.getUrl();
            } else {
                if (attachment != null) {
                    // For internal attachments, the src attribute in RenderAttachment is an absolute path, but without the domain name...
                    // When rendering for Word output, we need to prefix it with the full URL (CONF-5293)
                    if (context.getOutputType().equals(RenderContextOutputType.WORD)) {
                        // Grab the context path
                        String contextPath = context.getSiteRoot();
                        String domain = context.getBaseUrl();

                        // The baseUrl contains the context path, so strip it off (if it exists)
                        // This is quite dodgy, but fiddling with the URL passed into the RenderAttachment is going to be troublesome
                        if (contextPath != null && contextPath.length() != 0 && domain.indexOf(contextPath) != -1)
                            domain = domain.substring(0, domain.indexOf(contextPath));

                        imageUrl += domain;
                    }

                    imageUrl += attachment.getSrc();
                }
            }

            try {
                token = context.addRenderedContent(writeImage("<img src=\"" + HtmlEscaper.escapeAll(imageUrl, true) + "\" " + outputParameters(imageParams) + "/>",
                        imageParams, context, resource));
            } catch (MacroParameterValidationException ex) {
                return context.addRenderedContent(RenderUtils.error(ex.getMessage()));
            }
        }

        return token;
    }

    protected RendererAttachment getAttachment(RenderContext context, EmbeddedResource resource) {
        return attachmentManager.getAttachment(context, resource);
    }

    private String originalLink(EmbeddedResource resource) {
        return "!" + resource.getOriginalLinkText() + "!";
    }

    protected RendererAttachment getThumbnail(RendererAttachment attachment, RenderContext context, EmbeddedImage embeddedImage) {
        return attachmentManager.getThumbnail(attachment, context, embeddedImage);
    }

    private String generateThumbnail(Map imageParams, RendererAttachment attachment, RenderContext context, EmbeddedImage embeddedImage, EmbeddedResource resource) {
        if (attachment != null && TextUtils.stringSet(attachment.getComment()) && !imageParams.containsKey("title") && !imageParams.containsKey("TITLE")) {
            imageParams.put("title", attachment.getComment());
        }

        RendererAttachment thumb = null;
        if (attachment != null) {
            try {
                thumb = getThumbnail(attachment, context, embeddedImage);
            } catch (RuntimeException re) {
                return context.addRenderedContent(RenderUtils.error(re.getMessage()));
            }
        }

        try {
            if (thumb != null) {
                return writeImage(thumb.wrapGeneratedElement("<img src=\"" + HtmlEscaper.escapeAll(thumb.getSrc(), true) + "\" " + outputParameters(imageParams) + "role=\"presentation\"/>"),
                        imageParams, context, resource);
            } else {
                // even if the attachment is invalid, we still need to create some markup for it
                return writeImage("<img " + outputParameters(imageParams) + "role=\"presentation\"/>", imageParams, context, resource);
            }
        } catch (MacroParameterValidationException ex) {
            return context.addRenderedContent(RenderUtils.error(ex.getMessage()));
        }
    }

    protected String writeImage(String imageTag, Map<Object, Object> imageParams, RenderContext context, EmbeddedResource resource) {
        StringBuffer result = new StringBuffer();

        if (!context.isRenderingForWysiwyg()) {
            result.append("<span class=\"image-wrap\"");
            if (imageParams.get("align") != null) {
                result.append(" style=\"");

                String alignmentParam = imageParams.get("align").toString();
                if (alignmentParam.equals("left"))
                    result.append("float: left");
                else if (alignmentParam.equals("right"))
                    result.append("float: right");
                else if (alignmentParam.equals("center") || alignmentParam.equals("centre"))
                    result.append("display: block; text-align: center");

                result.append("\"");
            }
            result.append(">");
        }

        result.append(imageTag);

        if (!context.isRenderingForWysiwyg())
            result.append("</span>");

        return result.toString();
    }

    /**
     * @throws MacroParameterValidationException if any of the parameter values fail validation.
     */
    protected String outputParameters(Map params) throws MacroParameterValidationException {
        StringBuffer buff = new StringBuffer(20);
        StringBuffer cssbuff = new StringBuffer(30);

        SortedMap sortedParams = new TreeMap(params);

        for (Iterator iterator = sortedParams.keySet().iterator(); iterator.hasNext(); ) {
            String key = (String) iterator.next();

            // do not render the align parameter on the img tag
            if (key.equals("align") || key.equals(BORDERCOLOR_KEY))
                continue;

            // check if the attribute should be rendered as a CSS style attribute
            if (STYLE_ATTRIBUTES_MAPPING.containsKey(key)) {
                StyleInformation info = (StyleInformation) STYLE_ATTRIBUTES_MAPPING.get(key);
                String paramValue = (String) sortedParams.get(key);
                info.valueValidator.assertValid(paramValue);
                if (key.equals(BORDER_KEY)) {
                    if (!paramValue.contains("px") && !paramValue.contains("pt") && !paramValue.contains("em"))
                        paramValue += ("px");

                    paramValue += " solid ";
                    if (sortedParams.containsKey(BORDERCOLOR_KEY)) {
                        String borderColor = (String) sortedParams.get(BORDERCOLOR_KEY);
                        StyleInformation borderColorInfo = (StyleInformation) STYLE_ATTRIBUTES_MAPPING.get(BORDERCOLOR_KEY);
                        borderColorInfo.valueValidator.assertValid(borderColor);
                        paramValue += borderColor;
                    } else
                        paramValue += "black";
                }
                cssbuff.append(" ").append(info.cssStyleAttribute).append(": ").append(paramValue).append(";");
            } else {
                buff.append(HtmlEscaper.escapeAll(key, true)).append("=\"").
                        append(HtmlEscaper.escapeAll((String) sortedParams.get(key), true)).append("\" ");
            }
        }

        if (cssbuff.length() > 0) {
            // get rid of the trailing semi-colon
            cssbuff.deleteCharAt(cssbuff.length() - 1);

            // get rid of the preceding space
            if (cssbuff.charAt(0) == ' ')
                cssbuff.deleteCharAt(0);

            buff.append("style=\"").append(cssbuff).append("\" ");
        }

        return buff.toString();
    }
}
