package com.atlassian.renderer.wysiwyg.converter;

import java.util.HashMap;
import java.util.Map;

class TypeBasedSeparation implements Separation {
    /**
     * Separation objects define what seperator should go between a pair of sibling nodes.  It is used for the trip from xhtml to wiki markup.
     */
    private static final Map<NodePair, Separation> map = new HashMap<NodePair, Separation>();

    static {
        TypeBasedSeparation[] separators = new TypeBasedSeparation[]{
                new TypeBasedSeparation("img", "br", "\n", "\\\\\n "),  // CONF-13824, RNDR-50
                new TypeBasedSeparation("list", "heading", "\n", null),
                new TypeBasedSeparation("list", "p", "\n\n", null),
                new TypeBasedSeparation("list", "list", "\n\n", null),
                new TypeBasedSeparation("list", "div", "\n\n", null),
                new TypeBasedSeparation("list", "a", "\n\n", null),
                new TypeBasedSeparation("list", "text", "\n\n", "\n"), // when text follows a list in a table we can't correctly make
                // it  a new paragraph, we can only make it part of the last list item.
                new TypeBasedSeparation("list", "table", "\n\n", null),
                new TypeBasedSeparation("list", "br", "\n\n", "\\\\\n "),
                new TypeBasedSeparation("list", "userNewline", "\n", null),
                new TypeBasedSeparation("list", "li", "\n", "\n"),
                new TypeBasedSeparation("table", "table", "\n\n", null),
                new TypeBasedSeparation("table", "div", "\n", null),
                new TypeBasedSeparation("table", "a", "\n", null),
                new TypeBasedSeparation("table", "p", "\n", null),
                new TypeBasedSeparation("table", "userNewline", "\n", null),
                new TypeBasedSeparation("table", "br", "\n", "\\\\\n "),
                new TypeBasedSeparation("table", "forcedNewline", "\n", DefaultWysiwygConverter.TEXT_SEPARATOR),
                new TypeBasedSeparation("table", "text", "\n", null),
                new TypeBasedSeparation("table", "list", "\n", null),
                new TypeBasedSeparation("table", "heading", "\n", "", ""),
                new TypeBasedSeparation("text", "table", "\n", null),
                new TypeBasedSeparation("text", "list", "\n", "\n"),
                new TypeBasedSeparation("text", "forcedNewline", "\n", DefaultWysiwygConverter.TEXT_SEPARATOR),
                new TypeBasedSeparation("text", "heading", "\n", "", ""),
                new TypeBasedSeparation("text", "div", "\n", "\n", "\n"),
                new TypeBasedSeparation("text", "p", "\n", null, null),
                new TypeBasedSeparation("text", "br", "\n", "\\\\\n "),
                new TypeBasedSeparation("blockquote", "p", "\n", null),
                new TypeBasedSeparation("blockquote", "br", "\n", "\\\\\n "),
                new TypeBasedSeparation("p", "list", "\n", "\n"),
                new TypeBasedSeparation("p", "table", "\n", null),
                new TypeBasedSeparation("p", "text", "\n\n", null),
                new TypeBasedSeparation("p", "p", "\n\n", "\\\\\n", "\\\\\n"),
                new TypeBasedSeparation("p", "userNewline", "\n", "\\\\\n"),
                new TypeBasedSeparation("p", "heading", "\n", "", ""),
                new TypeBasedSeparation("p", "div", "\n", "", "\n"),
                new TypeBasedSeparation("p", "br", "\n", "\\\\\n "),
                new TypeBasedSeparation("heading", "table", "\n\n", null),
                new TypeBasedSeparation("heading", "div", "\n\n", "", ""),
                new TypeBasedSeparation("heading", "span", "\n\n", null, "\n"),
                new TypeBasedSeparation("heading", "text", "\n\n", null, "\n"),
                new TypeBasedSeparation("heading", "heading", "\n\n", null),
                new TypeBasedSeparation("heading", "p", "\n\n", "\n", "\n"),
                new TypeBasedSeparation("heading", "list", "\n\n", "\n", "\n"),
                new TypeBasedSeparation("heading", "br", "\n\n", "\\\\\n ", "\n"),
                new TypeBasedSeparation("heading", "userNewline", "\n", "\\\\\n "),
                new TypeBasedSeparation("tr", "tr", "\n", "\n"),
                new TypeBasedSeparation("a", "list", "\n", "\n"),
                new TypeBasedSeparation("a", "br", "\n", "\\\\\n "),
                new TypeBasedSeparation("br", "list", "", ""),
                new TypeBasedSeparation("br", "br", "\n", "\\\\\n"),
                new TypeBasedSeparation("div", "list", "\n", "\n"),
                new TypeBasedSeparation("div", "table", "\n", "\n"),
                new TypeBasedSeparation("div", "imagelink", "\n", "\n"),
                new TypeBasedSeparation("div", "text", "\n", "\n"),
                new TypeBasedSeparation("div", "p", "\n", null),
                new TypeBasedSeparation("div", "userNewline", "\n", null),
                new TypeBasedSeparation("div", "heading", "\n", "", ""),
                new TypeBasedSeparation("div", "div", "\n", "", "\n"),
                new TypeBasedSeparation("div", "br", "\n", "\\\\\n "),
                new TypeBasedSeparation("forcedNewline", "list", "\n", "\n"),
                new TypeBasedSeparation("forcedNewline", "li", "", ""),
                new TypeBasedSeparation("forcedNewline", "text", "\n", "\n"),
                new TypeBasedSeparation("forcedNewline", "forcedNewline", "\n", "\n"),
                new TypeBasedSeparation("forcedNewline", "a", "\n", "\n"),
                new TypeBasedSeparation("forcedNewline", "div", "\n", "\n"),
                new TypeBasedSeparation("forcedNewline", "br", "\n\n", "\n\\\\\n "),
                new TypeBasedSeparation("pre", "text", "\n", null),
                new TypeBasedSeparation("pre", "pre", "\n", null),
                new TypeBasedSeparation("pre", "br", "\n", "\\\\\n "),
                new TypeBasedSeparation("li", "li", "\n", "\n"),
                new TypeBasedSeparation("li", "list", "\n", "\n"),
                new TypeBasedSeparation("span", "heading", "\n", "", ""),
                new TypeBasedSeparation("span", "div", "\n", "\n", "\n"),
                new TypeBasedSeparation("span", "p", "\n", null, null),
                new TypeBasedSeparation("span", "br", "\n", "\\\\\n "),
                new TypeBasedSeparation("hr", "imagelink", "\n", "\n", "\n"),
                new TypeBasedSeparation("hr", "div", "\n", "\n", "\n"),
                new TypeBasedSeparation("hr", "list", "\n", "\n", "\n"),
                new TypeBasedSeparation("hr", "table", "\n", "\n", "\n"),
                new TypeBasedSeparation("hr", "p", "\n", null, null),
                new TypeBasedSeparation("hr", "text", "\n", null, null),
                new TypeBasedSeparation("span", "br", "\n", "\\\\\n "),
                new TypeBasedSeparation("font", "list", "\n", "\n", "\n"),
                new TypeBasedSeparation("font", "br", "\n", "\\\\\n "),
                new TypeBasedSeparation("emoticon", "text", " ", " "),
                new TypeBasedSeparation("emoticon", "br", "\n", "\\\\\n "),
                //new Separation("userNewline", "heading", "", ""),
        };
        for (TypeBasedSeparation separator : separators) {
            map.put(separator.getNodePair(), separator);
        }
    }

    /**
     * Returns a separation between two types.  Never returns null.
     *
     * @return a specific separation for two node types, or {@link Separation#ALWAYS_EMPTY} if none has
     * been specified
     */
    public static Separation getSeparation(String previous, String current) {
        Separation separation = map.get(new NodePair(previous, current));
        return separation != null ? separation : Separation.ALWAYS_EMPTY;
    }

    private final NodePair nodePair;
    private final String separator;
    private final String tableSeparator;
    private final String listSeparator;

    /**
     * Construct a definition of what separator should seperate two nodes of given types.
     *
     * @param previous       the first node in the pair
     * @param current        the second node in the pair
     * @param separator      the separator to place between the pair of nodes in standard contexts
     * @param tableSeparator the separator to place between the pair of nodes when within a table cell
     * @param listSeparator  the separator to place between the pair of nodes when within a list element
     */
    public TypeBasedSeparation(String previous, String current, String separator, String tableSeparator, String listSeparator) {
        this.nodePair = new NodePair(previous, current);
        this.separator = separator;
        this.tableSeparator = tableSeparator;
        this.listSeparator = listSeparator;
    }

    /**
     * Construct a definition of what separator should seperate two nodes of given types.  <code>separator</code> is used when within a list context.  If a different separator is required within lists, use the other constructor
     *
     * @param previous       the first node in the pair
     * @param current        the second node in the pair
     * @param separator      the separator to place between the pair of nodes in standard contexts
     * @param tableSeparator the separator to place between the pair of nodes when within a table cell
     */
    public TypeBasedSeparation(String previous, String current, String separator, String tableSeparator) {
        this.nodePair = new NodePair(previous, current);
        this.separator = separator;
        this.tableSeparator = tableSeparator;
        this.listSeparator = separator;
    }

    public String getSeparator() {
        return separator;
    }

    public String getTableSeparator() {
        return tableSeparator;
    }

    public String getListSeparator() {
        return listSeparator;
    }

    public NodePair getNodePair() {
        return nodePair;
    }

    public static class NodePair {
        private final String previous;
        private final String current;

        public NodePair(String previous, String current) {
            this.previous = previous;
            this.current = current;
        }

        public String getPrevious() {
            return previous;
        }

        public String getCurrent() {
            return current;
        }

        public boolean equals(Object obj) {
            if (obj instanceof NodePair) {
                NodePair otherNodePair = (NodePair) obj;
                return previous.equals(otherNodePair.previous) && current.equals(otherNodePair.current);
            }
            return false;
        }

        public int hashCode() {
            return 43 * (previous == null ? 1 : previous.hashCode()) + (current == null ? 1 : current.hashCode());
        }

    }

}
