package com.atlassian.renderer.wysiwyg.converter;

import com.atlassian.renderer.wysiwyg.NodeContext;
import org.apache.commons.lang.StringUtils;

/**
 * Converts a table cell, either th or td, to wiki markup.
 */
enum TableCellConverter implements Converter {
    TH("th", "||"), TD("td", "|");
    private final String nodeName;
    private final String delimiter;

    private TableCellConverter(String nodeName, String delimiter) {
        this.nodeName = nodeName;
        this.delimiter = delimiter;
    }

    public boolean canConvert(NodeContext nodeContext) {
        return nodeContext.hasNodeName(nodeName);
    }

    public String convertNode(NodeContext nodeContext, DefaultWysiwygConverter wysiwygConverter) {
        NodeContext childConversionContext = new NodeContext.Builder(nodeContext)
                .inTable(true).ignoreText(false).previousSibling(null).build();
        String convertedChildren = wysiwygConverter.convertChildren(childConversionContext);
        return delimiter + normaliseCellPadding(convertedChildren);
    }

    /**
     * The Conversion always pads table cell content with a space on either side.  All other whitespace padding is lost.
     * Empty cells are represented as a single space.
     */
    static String normaliseCellPadding(String s) {
        String trimmed = trimWhitespace(s);
        trimmed = trimmed.replaceAll("^&nbsp;", "");
        trimmed = trimmed.replaceAll("&nbsp;$", "");
        if (trimmed.equals("&nbsp;") || StringUtils.isEmpty(trimmed))
            return " ";

        return " " + trimmed + " ";
    }

    /**
     * Replaces any nonbreaking space characters with spaces, then trims leading and trailing whitespace.
     */
    private static String trimWhitespace(String s) {
        return s.replace((char) 160, (char) 32).trim();
    }
}
