package com.atlassian.renderer.wysiwyg.converter;

import com.atlassian.renderer.wysiwyg.ListContext;
import com.atlassian.renderer.wysiwyg.NodeContext;
import org.w3c.dom.Node;

/**
 * Converts a list by adding to the ListContext in the NodeContext and converting children.
 */
final class ListConverter implements Converter {
    static final ListConverter INSTANCE = new ListConverter();

    private ListConverter() {
    }

    public boolean canConvert(NodeContext nodeContext) {
        return nodeContext.hasNodeName("ol") || nodeContext.hasNodeName("ul");
    }

    public String convertNode(NodeContext nodeContext, DefaultWysiwygConverter wysiwygConverter) {
        String separator = wysiwygConverter.getSeparator("list", nodeContext);
        NodeContext.Builder builder = new NodeContext.Builder(nodeContext).ignoreText(true).previousSibling(null);
        String listType = getListType(nodeContext);

        if (isNewList(nodeContext.getNode().getParentNode()))
            builder.listContext(new ListContext(listType));
        else
            builder.listContext(new ListContext(listType, nodeContext.getListContext()));

        return separator + wysiwygConverter.convertChildren(builder.build()).trim();
    }

    /**
     * Determine the appropriate list type based on the attributes of the node.
     *
     * @return one of the constants in {@link com.atlassian.renderer.wysiwyg.ListContext}
     */
    private static String getListType(NodeContext nodeContext) {
        if (nodeContext.hasNodeName("ol"))
            return ListContext.NUMBERED;

        String typeAttribute = nodeContext.getAttribute("type");
        if (typeAttribute != null && typeAttribute.equals("square"))
            return ListContext.SQUARE;

        return ListContext.BULLETED;
    }


    /**
     * Determine whether we need to start a new list (CONF-7085).  The logic here is that lists can only be directly
     * nested into other lists, not into paragraphs or tables in lists.  This point seems arguable.
     *
     * @param parentNode - the parent node of the 'ul' node
     */
    private static boolean isNewList(Node parentNode) {
        if (parentNode == null)
            return false;
        String parentNodeName = parentNode.getNodeName().toLowerCase();
        //start new list as long as the parent is not a list item and is not a list (ul or ol)
        return !parentNodeName.equals("li") && !parentNodeName.equals("ul") && !parentNodeName.equals("ol");
    }
}
