package com.atlassian.renderer.wysiwyg.converter;

import com.atlassian.renderer.Icon;
import com.atlassian.renderer.IconManager;
import com.atlassian.renderer.wysiwyg.NodeContext;
import org.apache.commons.lang.StringUtils;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Converts an {@code <img>} tag to wiki markup.
 */
final class ImageConverter implements Converter {
    static ImageConverter INSTANCE = new ImageConverter();

    /**
     * Map of emoticon file names to emoticon symbols.
     *
     * @see #initEmoticonMap(IconManager)
     */
    private volatile Map<String, String> emoticonMap = null;

    private ImageConverter() {
    }

    public boolean canConvert(NodeContext nodeContext) {
        return nodeContext.hasNodeName("img");
    }

    public String convertNode(NodeContext nodeContext, DefaultWysiwygConverter wysiwygConverter) {
        // Images generated by the confluence renderer for wysiwyg have an "imagetext" attribute containing
        // the original markup of the image.
        if (nodeContext.getAttribute("imagetext") != null) {
            String imagetext = nodeContext.getAttribute("imagetext");

            // add image size params.  Note that we ignore the height if the width is not set.
            if (nodeContext.getAttribute("width") != null) {
                String params = "";
                // todo: the replace only replaces the digits.  There's a known bug for that.
                if (imagetext.contains("width="))
                    imagetext = imagetext.replaceAll("width=\\d*", "width=" + nodeContext.getAttribute("width"));
                else
                    params += "width=" + nodeContext.getAttribute("width");

                if (nodeContext.getAttribute("height") != null) {
                    if (imagetext.contains("height="))
                        imagetext = imagetext.replaceAll("height=\\d*", "height=" + nodeContext.getAttribute("height"));
                    else
                        params += ",height=" + nodeContext.getAttribute("height");
                }
                if (StringUtils.isNotEmpty(params)) {
                    if (!imagetext.contains("|"))
                        imagetext += "|" + params;
                    else
                        imagetext += "," + params;
                }
            }
            String separator = wysiwygConverter.getSeparator("imagelink", nodeContext);
            return separator + DefaultWysiwygConverter.TEXT_SEPARATOR + "!" + imagetext + "!";
        }
        if (nodeContext.getAttribute("src") != null) {
            String src = nodeContext.getAttribute("src");
            if (src.indexOf("images/icons/emoticons/") != -1) {
                return lookupEmoticonString(src.substring(src.indexOf("icons/emoticons/")), wysiwygConverter.getIconManager());
            }
        }
        return "";
    }


    private void initEmoticonMap(IconManager iconManager) {
        Map<String, String> newEmoticonMap = new HashMap<String, String>();
        List<String> symbolList = new ArrayList<String>(Arrays.asList(iconManager.getEmoticonSymbols()));
        // sort this so that the result is deterministic
        Collections.sort(symbolList);
        for (Object aSymbolList : symbolList) {
            String symbol = (String) aSymbolList;
            Icon icon = iconManager.getEmoticon(symbol);
            newEmoticonMap.put(icon.getPath(), symbol);
        }
        emoticonMap = newEmoticonMap;
    }

    private String lookupEmoticonString(String fileName, IconManager iconManager) {
        if (emoticonMap == null) {
            initEmoticonMap(iconManager);
        }
        String symbol = emoticonMap.get(fileName);
        if (symbol == null) {
            throw new RuntimeException("unrecognised emoticon " + fileName);
        } else {
            return symbol;
        }
    }

}
