package com.atlassian.renderer.wysiwyg.converter;

import com.atlassian.renderer.v2.macro.Macro;
import com.atlassian.renderer.wysiwyg.NodeContext;
import com.atlassian.renderer.wysiwyg.WysiwygMacroHelper;
import com.atlassian.renderer.wysiwyg.WysiwygNodeConverter;
import org.apache.commons.lang.StringUtils;

/**
 * Converts Divs to wikimarkup.  Divs can either be handled by passing on to the {@link WysiwygMacroHelper} or a specific
 * {@link WysiwygNodeConverter}.
 */
final class DivConverter implements Converter {
    static DivConverter INSTANCE = new DivConverter();

    private DivConverter() {
    }

    public boolean canConvert(NodeContext nodeContext) {
        return nodeContext.hasNodeName("div");
    }

    public String convertNode(NodeContext nodeContext, DefaultWysiwygConverter wysiwygConverter) {
        if (nodeContext.hasClass("error")) {
            // this div contains error text which must be discarded
            return "";
        }

        String macroName = WysiwygMacroHelper.getMacroName(nodeContext.getNode());
        if (macroName != null) {
            Macro macro = wysiwygConverter.getMacroManager().getEnabledMacro(macroName);
            NodeContext macroConversionContext = new NodeContext.Builder(nodeContext).ignoreText(false).build();
            String macroMarkup = WysiwygMacroHelper.convertMacroFromNode(macroConversionContext, wysiwygConverter, macro);
            if (StringUtils.isNotBlank(macroMarkup)) {
                return macroMarkup;
            }
        }
        if (WysiwygMacroHelper.isMacroBody(nodeContext.getNode())) {
            NodeContext conversionContext = new NodeContext.Builder(nodeContext).ignoreText(false).previousSibling(null).build();
            return wysiwygConverter.convertChildren(conversionContext);
        }

        // miscellaneous formatting divs
        // we are ignoring this div, so pass the current previousSibling as the initial previousSIbling of the contained block.
        NodeContext.Builder contextBuilder = new NodeContext.Builder(nodeContext).ignoreText(false);
        if (nodeContext.getPreviousSibling() == null)
            contextBuilder.previousSibling(nodeContext.getNode());
        return wysiwygConverter.convertChildren(contextBuilder.build());
    }
}
