package com.atlassian.renderer.wysiwyg;

import com.atlassian.renderer.wysiwyg.converter.DefaultWysiwygConverter;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;

/**
 * The context in which the node is being converted from xhtml to wiki markup.
 * <p>
 * This class is only used in the {@link WysiwygMacroHelper} at the moment and should be eventually
 * used in {@link DefaultWysiwygConverter} instead of passing so many parameters around.
 */
public class NodeContext {
    private final Node node;
    private Node previousSibling;
    private Styles styles;
    private final ListContext listContext;
    private final boolean inTable;
    private final boolean inListItem;
    private final boolean ignoreText;
    private final boolean escapeWikiMarkup;
    private final boolean inHeading;

    /**
     * @deprecated Since 6.0 use NodeContext#Builder to build NodeContexts instead.
     * This constructor does not allow setting the {@link #inHeading}.
     */
    public NodeContext(Node node, Node previousSibling, Styles styles, ListContext listContext, boolean inTable,
                       boolean inListItem, boolean ignoreText, boolean escapeWikiMarkup) {
        this.node = node;
        this.previousSibling = previousSibling;
        this.styles = styles;
        this.listContext = listContext;
        this.inTable = inTable;
        this.inListItem = inListItem;
        this.inHeading = false;
        this.ignoreText = ignoreText;
        this.escapeWikiMarkup = escapeWikiMarkup;
    }

    private NodeContext(Node node, Node previousSibling, Styles styles, ListContext listContext, boolean inTable,
                        boolean inListItem, boolean inHeading, boolean ignoreText, boolean escapeWikiMarkup) {
        this.node = node;
        this.previousSibling = previousSibling;
        this.styles = styles;
        this.listContext = listContext;
        this.inTable = inTable;
        this.inListItem = inListItem;
        this.inHeading = inHeading;
        this.ignoreText = ignoreText;
        this.escapeWikiMarkup = escapeWikiMarkup;
    }

    /**
     * @return a NodeContext for the first child node of this context. Returns null if
     * there is no first child.
     */
    public NodeContext getFirstChildNodeContext() {
        if (node.getFirstChild() == null)
            return null;

        return new Builder(this).node(node.getFirstChild()).previousSibling(null).build();
    }

    /**
     * @return a NodeContext for the first child node of this context. Returns null if
     * there is no first child. Copies the previous sibling down from the current nodeContext.
     */
    public NodeContext getFirstChildNodeContextPreservingPreviousSibling() {
        if (node.getFirstChild() == null)
            return null;

        return new Builder(this).node(node.getFirstChild()).build();
    }

    /**
     * Get a node context for the child of this context after the given one.  The new context will have most of the
     * settings of the parent context.
     *
     * @param child a childContext of this object.
     * @return a NodeContext for the next sibling node of the given child context. Returns null if
     * there is no next sibling.
     */
    public NodeContext getNodeContextForNextChild(NodeContext child) {
        if (child.getNode().getNextSibling() == null)
            return null;

        return new Builder(this).node(child.getNode().getNextSibling())
                .previousSibling(child.getNode()).build();
    }

    /**
     * Get a node context for the child after the given one, but keep the previous sibling set in the child context.
     * This is used to skip the given child in terms of separation detection.
     *
     * @param child a childContext of this object.
     * @return a NodeContext for the next sibling node of the given child context. Returns null if
     * there is no next sibling.
     */
    public NodeContext getNodeContextForNextChildPreservingPreviousSibling(NodeContext child) {
        if (child.getNode().getNextSibling() == null)
            return null;

        return new Builder(this).node(child.getNode().getNextSibling())
                .previousSibling(child.getPreviousSibling()).build();
    }

    public String invokeConvert(WysiwygNodeConverter wysiwygNodeConverter, WysiwygConverter wysiwygConverter) {
        return wysiwygNodeConverter.convertXHtmlToWikiMarkup(previousSibling, node, wysiwygConverter, styles, listContext, inTable, inListItem, ignoreText);
    }

    /**
     * Check the "class" attribute for a specific class
     *
     * @return true if the given class is found
     */
    public boolean hasClass(String className) {
        final String nodeClassesAttr = getAttribute("class");
        if (nodeClassesAttr == null)
            return false;
        String[] nodeClasses = nodeClassesAttr.split(" ");
        for (String nodeClass : nodeClasses) {
            if (nodeClass.equals(className))
                return true;
        }
        return false;
    }

    /**
     * Builder to help construct new NodeContexts.
     */
    public static class Builder {
        private Node node;
        private Node previousSibling;
        private Styles styles;
        private ListContext listContext;
        private boolean inTable;
        private boolean inListItem;
        private boolean inHeading;
        private boolean ignoreText;
        private boolean escapeWikiMarkup;

        /**
         * Start building a NodeContext with just a current Node. Other properties are set to
         * null (previousSibling), false or blank entries (Style and ListContext).
         */
        public Builder(Node node) {
            this.node = node;
            this.previousSibling = null;
            this.styles = new Styles();
        }

        public Builder(NodeContext originalNodeContext) {
            this.node = originalNodeContext.node;
            this.previousSibling = originalNodeContext.previousSibling;
            this.styles = new Styles(node, originalNodeContext.getStyles());
            this.listContext = originalNodeContext.listContext;
            this.inTable = originalNodeContext.inTable;
            this.inListItem = originalNodeContext.inListItem;
            this.inHeading = originalNodeContext.inHeading;
            this.ignoreText = originalNodeContext.ignoreText;
            this.escapeWikiMarkup = originalNodeContext.escapeWikiMarkup;
        }

        public NodeContext build() {
            return new NodeContext(node, previousSibling, styles, listContext, inTable,
                    inListItem, inHeading, ignoreText, escapeWikiMarkup);
        }

        public Builder node(Node node) {
            this.node = node;
            this.styles = new Styles(node, this.styles);
            return this;
        }

        public Builder previousSibling(Node previousSibling) {
            this.previousSibling = previousSibling;
            return this;
        }

        public Builder ignoreText(boolean ignoreText) {
            this.ignoreText = ignoreText;
            return this;
        }

        public Builder escapeWikiMarkup(boolean escapeWikiMarkup) {
            this.escapeWikiMarkup = escapeWikiMarkup;
            return this;
        }

        public Builder addStyle(String style) {
            this.styles = new Styles(style, this.styles);
            return this;
        }

        public Builder styles(Styles styles) {
            this.styles = styles;
            return this;
        }

        public Builder listContext(ListContext listContext) {
            this.listContext = listContext;
            return this;
        }

        public Builder inListItem(boolean inListItem) {
            this.inListItem = inListItem;
            return this;
        }

        public Builder inTable(boolean inTable) {
            this.inTable = inTable;
            return this;
        }

        public Builder inHeading(boolean inHeading) {
            this.inHeading = inHeading;
            return this;
        }
    }

    public Node getNode() {
        return node;
    }

    public Node getPreviousSibling() {
        return previousSibling;
    }

    public Styles getStyles() {
        return styles;
    }

    public ListContext getListContext() {
        return listContext;
    }

    public boolean isInTable() {
        return inTable;
    }

    public boolean isInListItem() {
        return inListItem;
    }

    public boolean isInHeading() {
        return inHeading;
    }

    public boolean isIgnoreText() {
        return ignoreText;
    }

    public boolean isEscapeWikiMarkup() {
        return escapeWikiMarkup;
    }

    /**
     * Retrieve the value of an attribute from the node as a boolean.  If the value is not
     * present, or not readily convertible to a boolean return the requested defaultValue.
     *
     * @param attributeName the attribute to find the value of
     * @param defaultValue  the value to return if no boolean value is found.
     * @return the value of an attribute
     */
    public boolean getBooleanAttributeValue(String attributeName, boolean defaultValue) {
        String attributeValue = getAttribute(attributeName);
        return String.valueOf(!defaultValue).equalsIgnoreCase(attributeValue) ? !defaultValue : defaultValue;
    }

    /**
     * Retrieve the attribute value from the node.
     *
     * @return the value of an attribute. Null if attribute isn't present.
     */
    public String getAttribute(String name) {
        NamedNodeMap map = node.getAttributes();
        if (map == null) {
            // this should never happen. Comment nodes have no attributes, but they should be filtered out
            // before getAttributes() is called
            return null;
        } else {
            Node n = node.getAttributes().getNamedItem(name);
            return n != null ? n.getNodeValue() : null;
        }
    }

    /**
     * Returns the name of the node in all lower case.
     *
     * @return lower case version of the node name.
     */
    public String getNodeName() {
        return node.getNodeName().toLowerCase();
    }

    /**
     * Return true if the node name matches the given string.  The check is case insensitive.
     */
    public boolean hasNodeName(String nodeName) {
        return node.getNodeName().equalsIgnoreCase(nodeName);
    }
}