package com.atlassian.renderer.v2.macro.basic.validator;

import java.util.HashMap;
import java.util.Map;

/**
 * Wraps a parameter map of {@literal String -> String} mappings, providing validation by means of
 * a mapping of parameters to {@link ParameterValidator}s.
 */
public class ValidatedMacroParameters {
    private final Map<String, String> parameters;
    private final Map<String, ParameterValidator> validators = new HashMap<String, ParameterValidator>();

    public ValidatedMacroParameters(Map<String, String> parameters) {
        this.parameters = parameters;
    }

    public void setValidator(String parameterName, ParameterValidator parameterValidator) {
        validators.put(parameterName, parameterValidator);
    }

    /**
     * Returns the value of the parameter from the underlying parameter map, or throws an exception
     * if the value does not pass the validation for that parameter configured by
     * {@link #setValidator(String, ParameterValidator)}. A parameter which does not have a validator configured
     * will return the value without validating it.
     *
     * @param parameterName the name of the parameter to retrieve
     * @return the parameter value, if it is valid or there is no validator for the parameter, or null if the parameter
     * is not specified.
     * @throws MacroParameterValidationException if the parameter is not valid
     */
    public String getValue(String parameterName) throws MacroParameterValidationException {
        String parameterValue = parameters.get(parameterName);
        if (parameterValue == null) {
            return null;
        }

        ParameterValidator validator = validators.get(parameterName);
        if (validator == null) {
            // if no validator, pass back unvalidated value
            return parameterValue;
        }

        validator.assertValid(parameterValue);
        return parameterValue;
    }
}
