package com.atlassian.renderer.v2.components.phrase;

import com.atlassian.renderer.RenderContext;
import com.atlassian.renderer.TokenType;
import com.atlassian.renderer.v2.RenderMode;
import com.atlassian.renderer.v2.Replacer;
import com.atlassian.renderer.v2.components.RendererComponent;

import java.util.HashMap;
import java.util.Map;
import java.util.regex.Pattern;

public class PhraseRendererComponent implements RendererComponent {
    private static final Replacer REPLACER_INS_TO_U = new Replacer(Pattern.compile("<(/?)ins>"), "<$1u>", "ins>");

    /**
     * Does a zero-width negative look-behind assertion for letters and digits.
     * That is, this expression can be placed before another expression to make a pattern
     * that will match the second expression as long as it isn't preceded immediately by
     * letters or numbers.
     */
    static final String NO_LETTERS_OR_DIGITS_BEFORE = "(?<![\\p{L}\\p{Nd}\\\\])";

    /**
     * Does a zero-width negative look-ahead assertion for letters and digits.
     * That is, this expression can be placed after another expression to make a pattern
     * that will match the second expression as long as it isn't followed immediately by
     * letters or numbers.
     */
    static final String NO_LETTERS_OR_DIGITS_AFTERWARDS = "(?![\\p{L}\\p{Nd}])";

    /**
     * Does a zero-width positive look-behind assertion for inline token strings.
     */
    static final String INLINE_TOKENS_BEFORE = "(?<=" + TokenType.INLINE.getTokenMarker() + ")";

    /**
     * Does a zero-width positive look-ahead assertion for inline token strings.
     */
    static final String INLINE_TOKENS_AFTERWARDS = "(?=" + TokenType.INLINE.getTokenMarker() + ")";


    /**
     * Does a zero-width negative look-behind assertion for letters and digits.
     * That is, this expression can be placed before another expression to make a pattern
     * that will match the second expression as long as it isn't preceded immediately by
     * letters or numbers.
     */
    public static final String VALID_START = "(" + NO_LETTERS_OR_DIGITS_BEFORE + "|" + INLINE_TOKENS_BEFORE + ")";

    /**
     * Pattern representing the valid content that can come after the end-delimiter.
     * That is:
     * <li>Characters that are not Letters or Digits</li>
     * <li>An inline token.</li>
     */
    public static final String VALID_END = "(" + NO_LETTERS_OR_DIGITS_AFTERWARDS + "|" + INLINE_TOKENS_AFTERWARDS + ")";

    private Replacer replacer;

    private static Map<String, PhraseRendererComponent> heresOneWePreparedEarlier = new HashMap<String, PhraseRendererComponent>();

    static {
        heresOneWePreparedEarlier.put("citation", new PhraseRendererComponent("\\?\\?", "cite"));
        heresOneWePreparedEarlier.put("strong", new PhraseRendererComponent("\\*", "b"));
        heresOneWePreparedEarlier.put("superscript", new PhraseRendererComponent("\\^", "sup"));
        heresOneWePreparedEarlier.put("subscript", new PhraseRendererComponent("~", "sub"));
        heresOneWePreparedEarlier.put("emphasis", new PhraseRendererComponent("_", "em"));
        heresOneWePreparedEarlier.put("deleted", new PhraseRendererComponent("-", "del"));
        heresOneWePreparedEarlier.put("inserted", new PhraseRendererComponent("\\+", "ins"));
        heresOneWePreparedEarlier.put("monospaced", new PhraseRendererComponent("\\{\\{", "\\}\\}", "tt"));
    }

    public static PhraseRendererComponent getDefaultRenderer(String name) {
        return heresOneWePreparedEarlier.get(name);
    }

    public PhraseRendererComponent(String delimiter, String tagName) {
        this(delimiter, delimiter, "<" + tagName + ">", "</" + tagName + ">");
    }

    public PhraseRendererComponent(String startDelimiter, String endDelimiter, String tagName) {
        this(startDelimiter, endDelimiter, "<" + tagName + ">", "</" + tagName + ">");
    }

    public PhraseRendererComponent(String startDelimiter, String endDelimiter, String startTag, String endTag) {
        this.replacer = new Replacer(
                makePattern(startDelimiter, endDelimiter),
                startTag + "$2" + endTag,
                startDelimiter.replaceAll("\\\\", ""), endDelimiter.replaceAll("\\\\", "")
        );
    }

    public String render(String wiki, RenderContext context) {
        String html = replacer.replaceAll(wiki);
        if (context.isRenderingForWysiwyg()) {
            html = REPLACER_INS_TO_U.replaceAll(html);
        }
        return html;
    }

    public boolean shouldRender(RenderMode renderMode) {
        return renderMode.renderPhrases();
    }


    /**
     * Creates a pattern that allows a start and end delimiter to be matched in a limited number
     * of circumstances.  A special case is that they will matched when within curly braces {}.
     * The pattern provides matching group 2 as the area between the delimiters.
     */
    static Pattern makePattern(String startDelimiter, String endDelimiter) {
        String startDelimiter2 = "\\{" + startDelimiter + "\\}";
        String endDelimiter2 = "\\{" + endDelimiter + "\\}";

        String phrase_content = "[^\\s" + startDelimiter + "]((?!" + endDelimiter + ")[\\p{L}\\p{Nd}\\p{Z}\\p{S}\\p{M}\\p{P}]*?[^\\s" + endDelimiter + "])??";

        return Pattern.compile(
                "(?:" +
                        "(?:" +
                        VALID_START +// $1
                        startDelimiter +
                        ")|" +
                        startDelimiter2 +
                        ")" +
                        "(" + phrase_content + ")" + //$2
                        "(?<!\\\\)" +
                        "(?:" +
                        "(?:" + endDelimiter +
                        VALID_END + //$3
                        ")|" + endDelimiter2 +
                        ")");
    }
}
