package com.atlassian.renderer.v2.components;

public final class HtmlEscaper {
    private static final String HTML_AMPERSAND = "&amp;";
    private static final String HTML_DOUBLE_QUOTE = "&quot;";
    private static final String HTML_GREATER_THAN = "&gt;";
    private static final String HTML_LEFT_SINGLE_QUOTE = "&lsquo;";
    private static final String HTML_LEFT_DOUBLE_QUOTE = "&ldquo;";
    private static final String HTML_LESS_THAN = "&lt;";
    private static final String HTML_RIGHT_SINGLE_QUOTE = "&rsquo;";
    private static final String HTML_RIGHT_DOUBLE_QUOTE = "&rdquo;";
    private static final String HTML_SINGLE_QUOTE = "&#39;";

    private static final CharMap ESCAPE_ALL_CHAR_MAP = new CharMap() {
        public String get(final char c) {
            switch (c) {
                case '\'':
                    return HTML_SINGLE_QUOTE;
                case '"':
                    return HTML_DOUBLE_QUOTE;
                case '&':
                    return HTML_AMPERSAND;
                case '>':
                    return HTML_GREATER_THAN;
                case '<':
                    return HTML_LESS_THAN;
                case 145: // Microsoft broken smart quote
                    return HTML_LEFT_SINGLE_QUOTE;
                case 146: // Microsoft broken smart quote
                    return HTML_RIGHT_SINGLE_QUOTE;
                case 147: // Microsoft broken smart quote
                    return HTML_LEFT_DOUBLE_QUOTE;
                case 148: // Microsoft broken smart quote
                    return HTML_RIGHT_DOUBLE_QUOTE;
                default:
                    return null;
            }
        }
    };

    private static final CharMap ESCAPE_ALL_EXCEPT_QUOTES_CHAR_MAP = new CharMap() {
        public String get(final char c) {
            switch (c) {
                case '&':
                    return HTML_AMPERSAND;
                case '>':
                    return HTML_GREATER_THAN;
                case '<':
                    return HTML_LESS_THAN;
                case 145: // Microsoft broken smart quote
                    return HTML_LEFT_SINGLE_QUOTE;
                case 146: // Microsoft broken smart quote
                    return HTML_RIGHT_SINGLE_QUOTE;
                case 147: // Microsoft broken smart quote
                    return HTML_LEFT_DOUBLE_QUOTE;
                case 148: // Microsoft broken smart quote
                    return HTML_RIGHT_DOUBLE_QUOTE;
                default:
                    return null;
            }
        }
    };
    private static final CharMap ESCAPE_AMPERSAND_CHAR_MAP = new CharMap() {
        public String get(char c) {
            return (c != '&') ? null : HTML_AMPERSAND;
        }
    };

    private HtmlEscaper() {
    }

    /**
     * Replaces the HTML special characters less-than (<code>&lt;</code>), greater-than (<code>&gt;</code>),
     * double-quote (<code>&quot;</code>), single-quote/apostrophe (<code>'</code>), and ampersand
     * (<code>&amp;</code>) with their equivalent entities in HTML 4 and returns the result.  Also
     * maps the Microsoft Windows-1252 "smart quotes" characters (145-148) with their equivalent
     * HTML entities.
     *
     * @param s                        the String to escape
     * @param preserveExistingEntities if <code>true</code>, will avoid escaping the ampersand in an existing
     *                                 entity like <code>&amp;lt;</code>. If <code>false</code>, the method will
     *                                 do a normal escaping by replace all matched characters.
     * @return the string with special characters replaced by entities.
     */
    public static String escapeAll(String s, boolean preserveExistingEntities) {
        return doReplacement(s, preserveExistingEntities, ESCAPE_ALL_CHAR_MAP);
    }

    /**
     * Does the same as {@link #escapeAll(String, boolean)}, except doesn't replace the ASCII quotation mark
     * characters double-quote (<code>"</code>) and single-quote/apostrophe (<code>'</code>).
     *
     * @param s                        the String to escape
     * @param preserveExistingEntities if <code>true</code>, will avoid escaping the ampersand in an existing
     *                                 entity like <code>&amp;lt;</code>. If <code>false</code>, the method will
     *                                 do a normal escaping by replace all matched characters.
     * @return the string with special characters replaced by entities.
     */
    public static String escapeAllExceptQuotes(String s, boolean preserveExistingEntities) {
        return doReplacement(s, preserveExistingEntities, ESCAPE_ALL_EXCEPT_QUOTES_CHAR_MAP);
    }

    /**
     * Replaces ampersand (<code>&amp;</code>) characters with the equivalent HTML entity
     * (<code>&amp;amp;</code>) and returns the result.
     *
     * @param s                        the String to escape
     * @param preserveExistingEntities if <code>true</code>, will avoid escaping the ampersand in an existing
     *                                 entity like <code>&amp;lt;</code>. If <code>false</code>, the method will
     *                                 do a normal escaping by replace all matched characters.
     * @return the string with special characters replaced by entities.
     */
    public static String escapeAmpersands(String s, boolean preserveExistingEntities) {
        return doReplacement(s, preserveExistingEntities, ESCAPE_AMPERSAND_CHAR_MAP);
    }

    /**
     * This method scans the input string <code>s</code> for the first character that needs to
     * be replaced.  If it does not find one, then <code>s</code> is returned without modification.
     * Otherwise, it delegates to <code>escapeWithPreserveHeavy</code> or <code>escapeWithoutPreserveHeavy</code>,
     * as appropriate.
     *
     * @param s                        the input string (may be <code>null</code>)
     * @param preserveExistingEntities <code>true</code> to protect entities like "<code>&amp;gt;</code>"
     *                                 from double-escaping to "<code>&amp;amp;gt;</code>"; <code>false</code>
     *                                 to escape all ampersands, regardless of context.
     * @param charMap                  the character replacement map to use
     * @return the escaped string
     */
    private static String doReplacement(final String s, final boolean preserveExistingEntities, final CharMap charMap) {
        if (s == null) {
            return null;
        }
        final int len = s.length();
        for (int i = 0; i < len; i++) {
            final char c = s.charAt(i);
            final String replacement = charMap.get(c);
            if (replacement != null) {
                if (!preserveExistingEntities) {
                    return escapeWithoutPreserveHeavy(s, charMap, i, replacement);
                }
                if (c != '&' || !entityAt(s, i)) {
                    return escapeWithPreserveHeavy(s, charMap, i, replacement);
                }
            }
        }
        return s;
    }

    /**
     * The heavy implementation that is called when the first replacement is found and
     * <code>preserveExistingEntities</code> is <code>false</code>.
     *
     * @param s           the original string (must not be <code>null</code>)
     * @param charMap     the character replacement map to use
     * @param i           the index of the first character to be escaped
     * @param replacement the replacement string for <code>s.charAt(i)</code>
     * @return the escaped string
     */
    private static String escapeWithoutPreserveHeavy(final String s, final CharMap charMap, int i, String replacement) {
        final int len = s.length();
        final StringBuilder sb = new StringBuilder(len + 64).append(s, 0, i).append(replacement);
        while (++i < len) {
            final char c = s.charAt(i);
            replacement = charMap.get(c);
            if (replacement != null) {
                sb.append(replacement);
            } else {
                sb.append(c);
            }
        }
        return sb.toString();
    }

    /**
     * The heavy implementation that is called when the first replacement is found and
     * <code>preserveExistingEntities</code> is <code>true</code>.
     *
     * @param s           the original string (must not be <code>null</code>)
     * @param charMap     the character replacement map to use
     * @param i           the index of the first character to be escaped
     * @param replacement the replacement string for <code>s.charAt(i)</code>
     * @return the escaped string
     */
    private static String escapeWithPreserveHeavy(final String s, final CharMap charMap, int i, String replacement) {
        final int len = s.length();
        final StringBuilder sb = new StringBuilder(len + 64).append(s, 0, i).append(replacement);
        while (++i < len) {
            final char c = s.charAt(i);
            replacement = charMap.get(c);
            if (replacement != null && (c != '&' || !entityAt(s, i))) {
                sb.append(replacement);
            } else {
                sb.append(c);
            }
        }
        return sb.toString();
    }


    // Hard-coded matcher for /^&[#A-Za-z0-9][A-Za-z0-9]{1,7};/
    private static boolean entityAt(String s, int index) {
        final int stop = Math.min(s.length(), index + 10);
        // Check minimum length
        if (index + 4 > stop) {
            return false;
        }

        // Initial '&' was checked by caller.
        // Next char must be either '#' or alphanumeric
        char c = s.charAt(++index);
        if (c != '#' && !isAlnum(c)) {
            return false;
        }

        // Next char must be alphanumeric
        if (!isAlnum(s.charAt(++index))) {
            return false;
        }

        // Then scan until we run out of room or terminate the {1,7} range
        while (++index < stop) {
            c = s.charAt(index);
            if (!isAlnum(c)) {
                return c == ';';
            }
        }

        // Hit the end of the range
        return false;
    }

    private static boolean isAlnum(final char c) {
        return (c >= 'A' && c <= 'Z') || (c >= 'a' && c <= 'z') || (c >= '0' && c <= '9');
    }

    static interface CharMap {
        String get(char c);
    }
}
