package com.atlassian.renderer.links;

import com.atlassian.renderer.util.RendererProperties;
import com.atlassian.renderer.v2.components.HtmlEscaper;

import java.util.Arrays;

public class UrlLink extends BaseLink {
    public static final String EXTERNAL_ICON = "external";
    public static final String MAILTO_ICON = "mailto";
    private final String unencodedUrl;

    public UrlLink(String url, String linkBody) {
        this(new GenericLinkParser(url));
        this.linkBody = linkBody;
    }

    public UrlLink(GenericLinkParser parser) {
        super(parser);
        iconName = EXTERNAL_ICON;
        url = parser.getNotLinkBody();
        setI18nTitle(RendererProperties.URL_LINK_TITLE, null);

        if (url.startsWith("///")) {
            url = url.substring(2);
            relativeUrl = true;
            setI18nTitle(RendererProperties.SITE_RELATIVE_LINK_TITLE, null);
        } else if (url.startsWith("//")) {
            url = url.substring(1);
            setI18nTitle(RendererProperties.RELATIVE_LINK_TITLE, null);
        }

        if (url.startsWith("\\\\"))
            url = "file:" + url.replaceAll("\\\\", "/");

        this.unencodedUrl = url;
        this.url = HtmlEscaper.escapeAll(this.url, true); // escape to avoid xss attacks

        if (url.startsWith("mailto:")) {
            // remove the "mailto:" from mail link bodies so we just get the
            // email address
            if (parser.getLinkBody() == null)
                linkBody = linkBody.substring(7);
            setI18nTitle(RendererProperties.SEND_MAIL_TO, Arrays.asList(linkBody));
            iconName = MAILTO_ICON;
        }
    }

    public String getLinkAttributes() {
        return " class=\"external-link\"";
    }

    /**
     * Vulnerable to XSS attacks unless escaped elsewhere. Use getUrl() to retrieve an escaped version of the URL.
     *
     * @return an unencoded URL that is vulnerable to XSS attacks unless escaped elsewhere.
     */
    public String getUnencodedUrl() {
        return unencodedUrl;
    }
}
