package com.atlassian.renderer;

import com.atlassian.renderer.v2.Renderable;
import org.apache.log4j.Category;

import java.util.ArrayList;
import java.util.List;

/**
 * Hides content from the renderer so that it will not be processed any further.
 * <p/>
 * <p>The RenderedContentStore associates blocks of rendered content (or Renderable objects that can later be
 * transformed into rendered content) with tokens. These tokens can then be substituted for the content blocks
 * during the rendering process. At the end of the chain of renderers, all tokens are replaced once more
 * with their associated content blocks.
 * <p/>
 * <p>Using this technique, it is possible to 'hide' content from the renderer, making it unnecessary to perform
 * nasty hacks to prevent the output of macros or renderer components from being broken by subsequent rendering
 * steps.
 */
public class RenderedContentStore {
    private static Category log = Category.getInstance(RenderedContentStore.class);

    /**
     * An unlikely-to-appear-in-real-text token, for use in block-replacement.
     *
     * @deprecated use {@link TokenType#BLOCK}.{@link TokenType#getTokenMarker() getTokenMarker}()
     */
    public static final String BLOCK_TOKEN = TokenType.BLOCK.getTokenMarker();

    /**
     * An unlikely-to-appear-in-real-text token, for use in inline content replacement.
     *
     * @deprecated use {@link TokenType#INLINE}.{@link TokenType#getTokenMarker() getTokenMarker}()
     */
    public static final String INLINE_TOKEN = TokenType.INLINE.getTokenMarker();

    // see also TokenType.INLINE_BLOCK.getTokenMarker

    /**
     * The key under which the store is placed in the render context
     */
    public static final String MAP_KEY = "RenderedContentStore";

    private List<Object> store = new ArrayList<Object>();

    /**
     * Retrieve the store from the render context. If the store is not in the context, it
     * is created.
     *
     * @param renderContext the render context to retrieve the store from
     * @return the active store for this context
     */
    public static RenderedContentStore getFromRenderContext(RenderContext renderContext) {
        return renderContext.getRenderedContentStore();
    }

    /**
     * Add content to the store. Returns a token that can be placed in
     * the page to be looked up later.
     *
     * @param content the content to add to the store
     * @param type    whether the content is {@link TokenType#INLINE} or {@link TokenType#BLOCK}. The token returned will
     *                reflect this type.
     * @return a token that the ContentReplaceFilter will be able to use to locate
     * the content at a later date.
     */
    public String addContent(Object content, TokenType type) {
        if (!(content instanceof String) && !(content instanceof Renderable)) {
            throw new RuntimeException("You can only store String and Renderable objects.");
        }

        store.add(content);
        return type.getTokenMarker() + (store.size() - 1) + type.getTokenMarker();
    }

    /**
     * Put some content block into the store. Returns a token that can be placed in
     * the page to be looked up later. The renderer knows not to draw things like
     * paragraphs around another block.
     *
     * @param content the content to add to the store
     * @return a token that the ContentReplaceFilter will be able to use to locate
     * the content at a later date.
     */
    public String addBlock(Object content) {
        return addContent(content, TokenType.BLOCK);
    }

    /**
     * Put some inline content into the store. Returns a token that can be placed in
     * the page to be looked up later. Inline elements will be surrounded by paragraphs
     * and have newlines inserted between them as necessary.
     *
     * @param content the content to add to the store
     * @return a token that the ContentReplaceFilter will be able to use to locate
     * the content at a later date.
     */
    public String addInline(Object content) {
        return addContent(content, TokenType.INLINE);
    }

    /**
     * Get some content back, given the entire token that was returned from a previous 'add'
     *
     * @param token a token that was returned from a previous add
     * @return the appropriate content, or null if the content could not be found for that
     * token.
     */
    public Object get(String token) {
        try {
            if (token == null || token.length() < 3) {
                log.warn("Could not find stored token: the token was null or too short. A filter or macro may be broken.");
                return null;
            }

            // todo: Fix this messiness.  The int needs to be parseable from the tokens, but this isn't the right way
            // to do it.  At least make sure the TokenType enforces that all TokenTypes have the same length somehow.
            final int len = TokenType.BLOCK.getTokenMarker().length();
            final int id = Integer.parseInt(token.substring(len, token.length() - len));
            return store.get(id);
        } catch (Exception e) {
            log.warn("Could not find stored token. A filter or macro may be broken. Exception: " + e.getMessage());
            return null;
        }
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof RenderedContentStore)) {
            return false;
        }

        final RenderedContentStore renderedContentStore = (RenderedContentStore) o;
        return (store == null) ? (renderedContentStore.store == null) : store.equals(renderedContentStore.store);
    }

    public int hashCode() {
        return (store != null ? store.hashCode() : 0);
    }

    public static String stripTokens(String text) {
        for (TokenType tokenType : TokenType.values()) {
            text = tokenType.getTokenDeleteReplacer().replaceAllSkippingConstantsCheck(text);
        }
        return text;
    }
}
