package com.atlassian.refapp.ctk.sal;

import com.atlassian.fugue.Option;
import com.atlassian.functest.junit.SpringAwareTestCase;
import com.atlassian.sal.api.user.UserKey;
import com.atlassian.sal.api.user.UserManager;
import com.atlassian.sal.api.user.UserProfile;
import com.atlassian.sal.api.usersettings.UserSettings;
import com.atlassian.sal.api.usersettings.UserSettingsBuilder;
import com.atlassian.sal.api.usersettings.UserSettingsService;
import com.google.common.base.Function;
import org.apache.commons.lang.StringUtils;
import org.junit.Before;
import org.junit.Test;

import javax.annotation.Nullable;

import static com.atlassian.sal.api.usersettings.UserSettingsService.MAX_KEY_LENGTH;
import static com.atlassian.sal.api.usersettings.UserSettingsService.MAX_STRING_VALUE_LENGTH;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

public class UserSettingsTest extends SpringAwareTestCase
{
    public static final String USERNAME = "admin";
    public static final String NOT_A_USERNAME = "notauser";
    public static final UserKey NOT_A_USER_KEY = new UserKey("notauser");
    public static final String KEY_1 = "key1";
    public static final String STRING_VALUE = "Hello";
    public static final String TOO_LONG_STRING = StringUtils.repeat("0", MAX_STRING_VALUE_LENGTH + 1);
    public static final String MAX_LENGTH_VALUE_STRING = TOO_LONG_STRING.substring(0, TOO_LONG_STRING.length() - 1);
    public static final String MAX_LENGTH_KEY_STRING = StringUtils.repeat("a", MAX_KEY_LENGTH);
    public static final long LONG_VALUE = 100l;
    private UserSettingsService userSettingsService;
    private UserManager userManager;
    private UserKey userKey;

    public void setService(UserSettingsService userSettingsService)
    {
        this.userSettingsService = userSettingsService;
    }

    public void setUserManager(UserManager userManager)
    {
        this.userManager = userManager;
    }

    private UserKey getUserKey(String userName)
    {
        UserProfile profile = userManager.getUserProfile(userName);
        return (profile != null) ? profile.getUserKey() : null;
    }

    @Before
    public void setUp()
    {
        userKey = getUserKey(USERNAME);
        // clear for user
        userSettingsService.updateUserSettings(userKey, new Function<UserSettingsBuilder, UserSettings>()
        {
            @Override
            public UserSettings apply(@Nullable UserSettingsBuilder input)
            {
                for (String key : input.getKeys())
                {
                    input.remove(key);
                }

                return input.build();
            }
        });
    }

    @Test
    public void userSettingsServiceShouldBeAvailable()
    {
        assertNotNull("UserSettingsService should be available to plugins", userSettingsService);
    }

    @Test(expected = IllegalArgumentException.class)
    public void noUserGetFailsWithCorrectException()
    {
        userSettingsService.getUserSettings(NOT_A_USERNAME);
    }

    @Test(expected = IllegalArgumentException.class)
    public void noUserProfileGetFailsWithCorrectException()
    {
        userSettingsService.getUserSettings(NOT_A_USER_KEY);
    }

    @Test(expected = IllegalArgumentException.class)
    public void noUserUpdateFailsWithCorrectException()
    {
        userSettingsService.updateUserSettings(NOT_A_USERNAME, new Function<UserSettingsBuilder, UserSettings>()
        {
            @Override
            public UserSettings apply(@Nullable UserSettingsBuilder input)
            {
                // should not get here
                fail("updateUserSettings() should not call the update function when the username is null");
                return input.build();
            }
        });
    }

    @Test(expected = IllegalArgumentException.class)
    public void noUserProfileUpdateFailsWithCorrectException()
    {
        userSettingsService.updateUserSettings(NOT_A_USER_KEY, new Function<UserSettingsBuilder, UserSettings>()
        {
            @Override
            public UserSettings apply(@Nullable UserSettingsBuilder input)
            {
                // should not get here
                fail("updateUserSettings() should not call the update function when the user profile is null");
                return input.build();
            }
        });
    }

    @Test
    public void typedBooleanUpdateStoresOnlyBooleanType()
    {
        // first insert a boolean
        userSettingsService.updateUserSettings(USERNAME, new Function<UserSettingsBuilder, UserSettings>()
        {
            @Override
            public UserSettings apply(UserSettingsBuilder input)
            {
                input.put(KEY_1, true);
                return input.build();
            }
        });

        // check that the stored value is there
        UserSettings userSettings = userSettingsService.getUserSettings(USERNAME);
        checkBooleanUpdateValue(userSettings);
    }

    @Test
    public void typedBooleanUpdateByProfileStoresOnlyBooleanType()
    {
        // first insert a boolean
        userSettingsService.updateUserSettings(userKey, new Function<UserSettingsBuilder, UserSettings>()
        {
            @Override
            public UserSettings apply(UserSettingsBuilder input)
            {
                input.put(KEY_1, true);
                return input.build();
            }
        });

        // check that the stored value is there
        UserSettings userSettings = userSettingsService.getUserSettings(userKey);
        checkBooleanUpdateValue(userSettings);
    }

    private void checkBooleanUpdateValue(UserSettings userSettings)
    {
        Option<Boolean> booleanValue = userSettings.getBoolean(KEY_1);
        assertFalse("boolean value should be present", booleanValue.isEmpty());
        assertTrue("boolean value should be true", booleanValue.get());

        // check the other value types are empty
        Option<String> stringValue = userSettings.getString(KEY_1);
        assertTrue("there should be no string type", stringValue.isEmpty());
        Option<Long> longValue = userSettings.getLong(KEY_1);
        assertTrue("there should be no long type", longValue.isEmpty());
    }

    @Test
    public void typedStringUpdateStoresOnlyStringType()
    {
        // now test a string overriding that boolean
        userSettingsService.updateUserSettings(USERNAME, new Function<UserSettingsBuilder, UserSettings>()
        {
            @Override
            public UserSettings apply(UserSettingsBuilder input)
            {
                input.put(KEY_1, STRING_VALUE);
                return input.build();
            }
        });

        // check that the stored value is there
        UserSettings userSettings = userSettingsService.getUserSettings(USERNAME);
        checkStringUpdateValue(userSettings);
    }

    @Test
    public void typedStringUpdateByProfileStoresOnlyStringType()
    {
        // now test a string overriding that boolean
        userSettingsService.updateUserSettings(userKey, new Function<UserSettingsBuilder, UserSettings>()
        {
            @Override
            public UserSettings apply(UserSettingsBuilder input)
            {
                input.put(KEY_1, STRING_VALUE);
                return input.build();
            }
        });

        // check that the stored value is there
        UserSettings userSettings = userSettingsService.getUserSettings(userKey);
        checkStringUpdateValue(userSettings);
    }

    private void checkStringUpdateValue(UserSettings userSettings)
    {
        Option<String> stringValue = userSettings.getString(KEY_1);
        assertFalse(stringValue.isEmpty());
        assertEquals(STRING_VALUE, stringValue.get());

        // check the other value types are empty
        Option<Boolean> booleanValue = userSettings.getBoolean(KEY_1);
        assertTrue(booleanValue.isEmpty());
        Option<Long> longValue = userSettings.getLong(KEY_1);
        assertTrue(longValue.isEmpty());
    }

    @Test
    public void typedLongUpdateStoresOnlyLongType()
    {
        // now test a long
        userSettingsService.updateUserSettings(USERNAME, new Function<UserSettingsBuilder, UserSettings>()
        {
            @Override
            public UserSettings apply(UserSettingsBuilder input)
            {
                input.put(KEY_1, LONG_VALUE);
                return input.build();
            }
        });

        // check that the stored value is there
        UserSettings userSettings = userSettingsService.getUserSettings(USERNAME);
        checkLongUpdateValue(userSettings);
    }

    @Test
    public void typedLongUpdateByProfileStoresOnlyLongType()
    {
        // now test a long
        userSettingsService.updateUserSettings(userKey, new Function<UserSettingsBuilder, UserSettings>()
        {
            @Override
            public UserSettings apply(UserSettingsBuilder input)
            {
                input.put(KEY_1, LONG_VALUE);
                return input.build();
            }
        });

        // check that the stored value is there
        UserSettings userSettings = userSettingsService.getUserSettings(userKey);
        checkLongUpdateValue(userSettings);
    }

    private void checkLongUpdateValue(UserSettings userSettings)
    {
        Option<Long> longValue = userSettings.getLong(KEY_1);
        assertFalse(longValue.isEmpty());
        assertEquals(LONG_VALUE, (long) longValue.get());

        // check the other value types are empty
        Option<Boolean> booleanValue = userSettings.getBoolean(KEY_1);
        assertTrue(booleanValue.isEmpty());
        Option<String> stringValue = userSettings.getString(KEY_1);
        assertTrue(stringValue.isEmpty());
    }

    @Test
    public void updatingUserSettingsWithMaxLengthKeySucceeds()
    {
        userSettingsService.updateUserSettings(USERNAME, new Function<UserSettingsBuilder, UserSettings>()
        {
            @Override
            public UserSettings apply(UserSettingsBuilder input)
            {
                input.put(MAX_LENGTH_KEY_STRING, true);
                return input.build();
            }
        });
    }

    @Test
    public void updatingUserSettingsByProfileWithMaxLengthKeySucceeds()
    {
        userSettingsService.updateUserSettings(userKey, new Function<UserSettingsBuilder, UserSettings>()
        {
            @Override
            public UserSettings apply(UserSettingsBuilder input)
            {
                input.put(MAX_LENGTH_KEY_STRING, true);
                return input.build();
            }
        });
    }

    @Test(expected = IllegalArgumentException.class)
    public void updatingUserSettingsWithTooLongKeyFails()
    {
        userSettingsService.updateUserSettings(USERNAME, new Function<UserSettingsBuilder, UserSettings>()
        {
            @Override
            public UserSettings apply(UserSettingsBuilder input)
            {
                input.put(TOO_LONG_STRING, true);
                return input.build();
            }
        });
    }

    @Test(expected = IllegalArgumentException.class)
    public void updatingUserSettingsByProfileWithTooLongKeyFails()
    {
        userSettingsService.updateUserSettings(userKey, new Function<UserSettingsBuilder, UserSettings>()
        {
            @Override
            public UserSettings apply(UserSettingsBuilder input)
            {
                input.put(TOO_LONG_STRING, true);
                return input.build();
            }
        });
    }

    @Test
    public void updatingUserSettingsWithMaxLengthStringValueSucceeds()
    {
        userSettingsService.updateUserSettings(USERNAME, new Function<UserSettingsBuilder, UserSettings>()
        {
            @Override
            public UserSettings apply(UserSettingsBuilder input)
            {
                input.put(KEY_1, MAX_LENGTH_VALUE_STRING);
                return input.build();
            }
        });
    }

    @Test
    public void updatingUserSettingsByProfileWithMaxLengthStringValueSucceeds()
    {
        userSettingsService.updateUserSettings(userKey, new Function<UserSettingsBuilder, UserSettings>()
        {
            @Override
            public UserSettings apply(UserSettingsBuilder input)
            {
                input.put(KEY_1, MAX_LENGTH_VALUE_STRING);
                return input.build();
            }
        });
    }

    @Test(expected = IllegalArgumentException.class)
    public void updatingUserSettingsWithTooLongStringValueFails()
    {
        userSettingsService.updateUserSettings(USERNAME, new Function<UserSettingsBuilder, UserSettings>()
        {
            @Override
            public UserSettings apply(UserSettingsBuilder input)
            {
                input.put(KEY_1, TOO_LONG_STRING);
                return input.build();
            }
        });
    }

    @Test(expected = IllegalArgumentException.class)
    public void updatingUserSettingsByProfileWithTooLongStringValueFails()
    {
        userSettingsService.updateUserSettings(userKey, new Function<UserSettingsBuilder, UserSettings>()
        {
            @Override
            public UserSettings apply(UserSettingsBuilder input)
            {
                input.put(KEY_1, TOO_LONG_STRING);
                return input.build();
            }
        });
    }
}
