package com.atlassian.refapp.ctk.sal;

import com.atlassian.functest.junit.SpringAwareTestCase;
import com.atlassian.httpclient.api.HttpClient;
import com.atlassian.sal.api.ApplicationProperties;
import com.atlassian.sal.api.UrlMode;
import com.atlassian.sal.api.web.context.HttpContext;
import org.junit.Test;

import java.io.IOException;
import java.net.URL;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;

import static org.junit.Assert.assertTrue;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertEquals;

public class ApplicationPropertiesTest extends SpringAwareTestCase
{
    private ApplicationProperties appProp;

    private HttpContext httpContext;
    private HttpClient httpClient;

    public void setApplicationProperties(ApplicationProperties applicationProperties)
    {
        appProp = applicationProperties;
    }

    public void setHttpContext(HttpContext httpContext)
    {
        this.httpContext = httpContext;
    }

    public void setHttpClient(final HttpClient httpClient)
    {
        this.httpClient = httpClient;
    }

    @Test
    public void testApplicationPropertiesAvailable()
    {
        assertNotNull("ApplicationProperties should be available to plugins", appProp);
    }

    @Test
    public void testGetBuildDateNotNull()
    {
        assertNotNull(appProp.getBuildDate());
    }

    @Test
    public void testGetBuildNumberNotNull()
    {
        assertNotNull(appProp.getBuildNumber());
    }

    @Test
    public void testGetDisplayNameNotNull()
    {
        assertNotNull(appProp.getDisplayName());
    }

    @Test
    public void testGetVersionNotNull()
    {
        assertNotNull(appProp.getVersion());
    }

    @Test
    public void testGetBaseUrlNotNull()
    {
        assertNotNull(appProp.getBaseUrl());
    }

    @Test
    public void testGetBaseUrlAccessible() throws IOException
    {
        assertEquals("the base url must be accessible", 200, getUrlStatusCode(appProp.getBaseUrl()));
    }

    @Test
    public void testGetCanonicalBaseUrlAccessible() throws IOException
    {
        assertEquals("the CANONICAL base url must be accessible", 200, getUrlStatusCode(appProp.getBaseUrl(UrlMode.CANONICAL)));
    }

    @Test
    public void testGetAbsoluteBaseUrlAccessible() throws IOException
    {
        assertEquals("the ABSOLUTE base url must be accessible", 200, getUrlStatusCode(appProp.getBaseUrl(UrlMode.ABSOLUTE)));
    }

    @Test
    public void testGetRelativeBaseUrlAccessible() throws IOException
    {
        String contextPath = appProp.getBaseUrl(UrlMode.RELATIVE);
        String baseUrl = httpContext.getRequest().getRequestURL().toString();
        String url = new URL(new URL(baseUrl), contextPath).toString();
        assertEquals("the RELATIVE base url must be accessible", 200, getUrlStatusCode(url));
    }

    @Test
    public void testGetRelativeCanonicalBaseUrlAccessible() throws IOException
    {
        String contextPath = appProp.getBaseUrl(UrlMode.RELATIVE_CANONICAL);
        String baseUrl = httpContext.getRequest().getRequestURL().toString();
        String url = new URL(new URL(baseUrl), contextPath).toString();
        assertEquals("the RELATIVE_CANONICAL base url must be accessible", 200, getUrlStatusCode(url));
    }

    @Test
    public void testGetRelativeBaseUrlOnRequestThread() throws IOException
    {
        String contextPath = httpContext.getRequest().getContextPath();
        assertEquals("the RELATIVE base url must be the context path on request threads", contextPath,
            appProp.getBaseUrl(UrlMode.RELATIVE));
    }

    @Test
    public void testGetRelativeBaseUrlOnWorkerThread() throws Exception
    {
        String baseUrl = appProp.getBaseUrl(UrlMode.CANONICAL);
        URL url = new URL(baseUrl);
        final String contextPath = url.getPath();
        final ExecutorService executorService = Executors.newSingleThreadExecutor();
        try
        {
            final Future<String> result = executorService.submit(new Callable<String>() {

                @Override
                public String call() throws Exception
                {
                    return appProp.getBaseUrl(UrlMode.RELATIVE);
                }
            });
            assertEquals("the RELATIVE base url must be derived from CANONICAL on worker threads", contextPath, result.get());
        }
        finally
        {
            executorService.shutdown();
        }
    }
    @Test
    public void testGetAutoBaseUrlRelativeOnRequestThread() throws IOException
    {
        assertEquals("the AUTO base url must be RELATIVE on request threads", appProp.getBaseUrl(UrlMode.RELATIVE),
            appProp.getBaseUrl(UrlMode.AUTO));
    }

    @Test
    public void testGetAutoBaseUrlCanonicalOnWorkerThread() throws Exception
    {
        final ExecutorService executorService = Executors.newSingleThreadExecutor();
        try
        {
            final Future<String> result = executorService.submit(new Callable<String>() {

                @Override
                public String call() throws Exception
                {
                    return appProp.getBaseUrl(UrlMode.AUTO);
                }
            });
            assertEquals("the AUTO base url must be CANONICAL on worker threads", appProp.getBaseUrl(UrlMode.CANONICAL), result.get());
        }
        finally
        {
            executorService.shutdown();
        }
    }

    @Test
    public void testGetHomeDirectoryNotNull()
    {
        assertNotNull(appProp.getHomeDirectory());
    }

    @Test
    public void testHomeDirectoryMustExist()
    {
        assertTrue("the home directory must exist", appProp.getHomeDirectory().exists());
        assertTrue("it must be a directory", appProp.getHomeDirectory().isDirectory());
    }

    private int getUrlStatusCode(String url)
    {
        return httpClient.newRequest(url).get().claim().getStatusCode();
    }
}
