package com.atlassian.refapp.ctk.sal;

import com.atlassian.functest.junit.SpringAwareTestCase;
import com.atlassian.sal.api.executor.ThreadLocalDelegateExecutorFactory;
import com.atlassian.sal.api.web.context.HttpContext;
import org.junit.Test;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;

import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;

import static org.junit.Assert.*;

public class HttpContextTest extends SpringAwareTestCase
{
    private HttpContext httpContext;
    private ThreadLocalDelegateExecutorFactory threadLocalDelegateExecutorFactory;

    public void setHttpContext(HttpContext httpContext)
    {
        this.httpContext = httpContext;
    }

    public void setThreadLocalDelegateExecutorFactory(ThreadLocalDelegateExecutorFactory threadLocalDelegateExecutorFactory)
    {
        this.threadLocalDelegateExecutorFactory = threadLocalDelegateExecutorFactory;
    }

    @Test
    public void httpContextIsNonNullInRequestThread()
    {
        final HttpServletRequest request = httpContext.getRequest();
        final HttpServletResponse response = httpContext.getResponse();
        final HttpSession session = httpContext.getSession(true);
        assertNotNull("Request should be available from a HTTP thread", request);
        assertNotNull("Response should be available from a HTTP thread", response);
        assertNotNull("Session should be available from a HTTP thread", session);
        assertEquals("Session from context should not change", session.getId(), httpContext.getSession(false).getId());
        assertEquals("Session from request should not change", session.getId(), request.getSession(false).getId());
    }

    @Test
    public void httpContextIsNullInExecutorThread() throws ExecutionException, InterruptedException
    {
        final ExecutorService executorService = threadLocalDelegateExecutorFactory.createExecutorService(
            Executors.newSingleThreadExecutor());
        try
        {
            Future<HttpServletRequest> request = executorService.submit(new Callable<HttpServletRequest>()
            {
                @Override
                public HttpServletRequest call() throws Exception
                {
                    return httpContext.getRequest();
                }
            });
            Future<HttpServletResponse> response = executorService.submit(new Callable<HttpServletResponse>()
            {
                @Override
                public HttpServletResponse call() throws Exception
                {
                    return httpContext.getResponse();
                }
            });
            Future<HttpSession> sessionGet = executorService.submit(new Callable<HttpSession>()
            {
                @Override
                public HttpSession call() throws Exception
                {
                    return httpContext.getSession(false);
                }
            });
            Future<HttpSession> sessionCreate = executorService.submit(new Callable<HttpSession>()
            {
                @Override
                public HttpSession call() throws Exception
                {
                    return httpContext.getSession(false);
                }
            });
            assertNull("Request should not be available from a worker thread", request.get());
            assertNull("Response should not be available from a worker thread", response.get());
            assertNull("Session should not be available from a worker thread", sessionGet.get());
            assertNull("Session should not be creatable from a worker thread", sessionCreate.get());
        }
        finally
        {
            executorService.shutdown();
            executorService.awaitTermination(1, TimeUnit.SECONDS);
        }
    }
}
