package com.atlassian.util.profiling;

import com.atlassian.plugin.util.PluginKeyStack;
import com.atlassian.util.profiling.strategy.MetricStrategy;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import java.time.Duration;
import java.util.Arrays;
import java.util.Collection;
import java.util.LinkedHashSet;
import java.util.Objects;
import java.util.Set;

import static com.atlassian.util.profiling.MetricKey.metricKey;
import static com.atlassian.util.profiling.MetricTag.FROM_PLUGIN_KEY_TAG_KEY;
import static com.atlassian.util.profiling.MetricTag.INVOKER_PLUGIN_KEY_TAG_KEY;
import static com.atlassian.util.profiling.MetricTag.SEND_ANALYTICS;
import static java.util.Arrays.asList;
import static java.util.Collections.emptySet;
import static java.util.Objects.isNull;
import static java.util.Objects.requireNonNull;

/**
 * Factory and utility methods for create {@link MetricTimer} and {@link Histogram}.
 *
 * <pre>
 *     try (Ticker ignored = Metrics.startTimer("my-timer")) {
 *         // monitored code block here
 *     }
 * </pre>
 *
 * @since 3.0
 */
@ParametersAreNonnullByDefault
public class Metrics {
    private static final MetricsConfiguration CONFIGURATION = new MetricsConfiguration();

    private Metrics() {
        throw new UnsupportedOperationException("Metrics is an utility class and should not be instantiated");
    }

    @Nonnull
    public static MetricsConfiguration getConfiguration() {
        return CONFIGURATION;
    }

    @Nonnull
    public static Histogram histogram(String name) {
        return metric(name).histogram();
    }

    /**
     * @since 3.4
     */
    @Nonnull
    public static Histogram histogram(String name, Collection<MetricTag> tags) {
        return metric(name).tags(tags).histogram();
    }

    /**
     * @since 3.4
     */
    @Nonnull
    public static Histogram histogram(String name, MetricTag... tags) {
        return metric(name).tags(tags).histogram();
    }

    @Nonnull
    public static MetricTimer timer(String name) {
        return new DefaultMetricTimer(metricKey(name));
    }

    /**
     * @since 3.4
     */
    @Nonnull
    public static MetricTimer timer(String name, Collection<MetricTag> tags) {
        return new DefaultMetricTimer(metricKey(name, tags));
    }

    /**
     * @since 3.4
     */
    @Nonnull
    public static MetricTimer timer(String name, MetricTag... tags) {
        return metric(name).tags(tags).timer();
    }

    /**
     * @since 3.5.0
     */
    @Nonnull
    public static LongRunningMetricTimer longRunningTimer(String name) {
        return longRunningTimer(name, emptySet());
    }

    /**
     * @since 3.5.0
     */
    @Nonnull
    public static LongRunningMetricTimer longRunningTimer(String name, Collection<MetricTag> tags) {
        return new DefaultLongRunningMetricTimer(metricKey(name, tags));
    }

    /**
     * @since 3.5.0
     */
    @Nonnull
    public static LongRunningMetricTimer longRunningTimer(String name, MetricTag... tags) {
        return longRunningTimer(name, Arrays.asList(tags));
    }

    @Nonnull
    public static Ticker startTimer(String name) {
        return startTimer(name, emptySet());
    }

    /**
     * @since 3.4
     */
    @Nonnull
    public static Ticker startTimer(String name, MetricTag... tags) {
        return startTimer(name, asList(tags));
    }

    /**
     * @since 3.4
     */
    @Nonnull
    public static Ticker startTimer(String name, Collection<MetricTag> tags) {
        return metric(name).tags(tags).startTimer();
    }

    /**
     * @param name metric name
     * @return metric builder
     * @since 3.5
     */
    public static Builder metric(String name){
        return new Builder(name);
    }

    /**
     * @since 3.5.0
     */
    public static class Builder {

        private final String name;
        private final Set<MetricTag> tags = new LinkedHashSet<>();

        protected Builder(String name) {
            this.name = name;
        }

        public Builder tag(@Nonnull String key, @Nullable String value) {
            return tags(MetricTag.of(key, value));
        }

        public Builder tags(MetricTag... tags) {
            return tags(asList(tags));
        }

        public Builder tags(Iterable<MetricTag> tags) {
            for (MetricTag tag: tags) {
                this.tags.add(tag);
            }
            return this;
        }

        /**
         * Attribute a plugin for providing a module that is measured
         * @since 3.6.0
         */
        public Builder fromPluginKey(@Nullable String pluginKey) {
            // There can only be one plugin key that the module comes from
            tags.stream()
                    .filter(tag -> Objects.equals(tag.getKey(), FROM_PLUGIN_KEY_TAG_KEY))
                    .findFirst()
                    .ifPresent(tags::remove);

            if (isNull(pluginKey)) {
                return this;
            }

            return tag(FROM_PLUGIN_KEY_TAG_KEY, pluginKey);
        }

        /**
         * Attribute the plugin for invoking measured code
         * @since 3.6.0
         */
        public Builder invokerPluginKey(@Nullable String pluginKey) {
            // There can only be one invoker
            tags.stream()
                    .filter(tag -> Objects.equals(tag.getKey(), INVOKER_PLUGIN_KEY_TAG_KEY))
                    .findFirst()
                    .ifPresent(tags::remove);

            if (isNull(pluginKey)) {
                return this;
            }

            return tag(INVOKER_PLUGIN_KEY_TAG_KEY, pluginKey);
        }

        /**
         * Attribute the plugin at the top of the call stack for invoking measured code
         * <p>
         * <strong>No guarantee</strong>, that the call stack will be able to be traversed, i.e if work is
         * delegated to another thread.
         * @since 3.6.0
         */
        public Builder withInvokerPluginKey() {
            return invokerPluginKey(PluginKeyStack.getFirstPluginKey());
        }

        public Builder withAnalytics() {
            return tags(SEND_ANALYTICS);
        }

        public Histogram histogram() {
            return new DefaultHistogram(metricKey(name, tags));
        }

        public MetricTimer timer() {
            return new DefaultMetricTimer(metricKey(name, tags));
        }

        /**
         * @since 3.6.0
         */
        public LongRunningMetricTimer longRunningTimer() {
            return new DefaultLongRunningMetricTimer(metricKey(name, tags));
        }

        public Ticker startTimer() {
            if (CONFIGURATION.isEnabled()) {
                return timer().start();
            }
            return Ticker.NO_OP;
        }

        /**
         * @since 3.6.0
         */
        public Ticker startLongRunningTimer() {
            if (CONFIGURATION.isEnabled()) {
                return longRunningTimer().start();
            }
            return Ticker.NO_OP;
        }
    }

    /**
     * @since 3.5.0
     */
    @Nonnull
    public static Ticker startLongRunningTimer(String name) {
        return startLongRunningTimer(name, emptySet());
    }

    /**
     * @since 3.5.0
     */
    @Nonnull
    public static Ticker startLongRunningTimer(String name, MetricTag... tags) {
        return startLongRunningTimer(name, asList(tags));
    }

    /**
     * @since 3.5.0
     */
    @Nonnull
    public static Ticker startLongRunningTimer(String name, Collection<MetricTag> tags) {
        if (CONFIGURATION.isEnabled()) {
            return longRunningTimer(name, tags).start();
        }
        return Ticker.NO_OP;
    }

    @ParametersAreNonnullByDefault
    private static class DefaultMetricTimer implements MetricTimer {
        private static final Logger log = LoggerFactory.getLogger(DefaultMetricTimer.class);

        private final MetricKey metricKey;

        DefaultMetricTimer(MetricKey name) {
            this.metricKey = requireNonNull(name, "metricKey");
        }

        @Nonnull
        @Override
        public Ticker start() {
            Collection<MetricStrategy> metricStrategies = StrategiesRegistry.getMetricStrategies();
            if (!getConfiguration().isEnabled() || metricStrategies.isEmpty()) {
                return Ticker.NO_OP;
            }

            CompositeTicker compositeTicker = null;
            for (MetricStrategy strategy : metricStrategies) {
                try {
                    compositeTicker = Tickers.addTicker(strategy.startTimer(metricKey), compositeTicker);
                } catch (RuntimeException e) {
                    log.warn("Failed to start metric trace for {}", metricKey, e);
                }
            }

            return compositeTicker == null ? Ticker.NO_OP : compositeTicker;
        }

        @Override
        public void update(Duration time) {
            if (getConfiguration().isEnabled()) {
                requireNonNull(time, "time");
                for (MetricStrategy strategy : StrategiesRegistry.getMetricStrategies()) {
                    try {
                        strategy.updateTimer(metricKey, time);
                    } catch (RuntimeException e) {
                        log.warn("Failed to update metric for {}", metricKey, e);
                    }
                }
            }
        }
    }

    /**
     * @since 3.5.0
     */
    @ParametersAreNonnullByDefault
    private static class DefaultLongRunningMetricTimer implements LongRunningMetricTimer {
        private static final Logger log = LoggerFactory.getLogger(DefaultLongRunningMetricTimer.class);

        private final MetricKey metricKey;

        DefaultLongRunningMetricTimer(MetricKey name) {
            this.metricKey = requireNonNull(name, "metricKey");
        }

        @Nonnull
        @Override
        public Ticker start() {
            Collection<MetricStrategy> metricStrategies = StrategiesRegistry.getMetricStrategies();
            if (!getConfiguration().isEnabled() || metricStrategies.isEmpty()) {
                return Ticker.NO_OP;
            }

            CompositeTicker compositeTicker = null;
            for (MetricStrategy strategy : metricStrategies) {
                try {
                    compositeTicker = Tickers.addTicker(strategy.startLongRunningTimer(metricKey), compositeTicker);
                } catch (RuntimeException e) {
                    log.warn("Failed to start metric trace for {}", metricKey, e);
                }
            }

            return compositeTicker == null ? Ticker.NO_OP : compositeTicker;
        }
    }

    private static class DefaultHistogram implements Histogram {
        private static final Logger log = LoggerFactory.getLogger(DefaultHistogram.class);

        private final MetricKey metricKey;

        private DefaultHistogram(MetricKey metricKey) {
            this.metricKey = requireNonNull(metricKey, "metricKey");
        }

        @Override
        public void update(long value) {
            if (getConfiguration().isEnabled()) {
                for (MetricStrategy strategy : StrategiesRegistry.getMetricStrategies()) {
                    try {
                        strategy.updateHistogram(metricKey, value);
                    } catch (RuntimeException e) {
                        log.warn("Failed to update histogram for {}", metricKey, e);
                    }
                }
            }
        }
    }
}
