package com.atlassian.util.profiling;

import javax.annotation.Nullable;
import java.util.HashSet;
import java.util.Set;

import static java.util.Collections.emptySet;

/**
 * Provides similar capability to <a href="http://logback.qos.ch/manual/mdc.html">MDC</a>,
 * i.e. for adding contextual tags onto all metrics generated during the enclosed execution context.
 *
 * E.g.
 * <pre>{@code
 * try (Closeable t = MetricTagContext.put(MetricTag.of("projectKey", projectKey))) {
 *     reindexIssue(issueId);
 * }
 * }</pre>
 */
public class MetricTagContext {
    private static InheritableThreadLocal<Set<MetricTag>> threadLocal = new InheritableThreadLocal<Set<MetricTag>>() {
        @Override
        protected Set<MetricTag> childValue(Set<MetricTag> parentValue) {
            return parentValue == null ? null : new HashSet(parentValue);
        }
    };

    public static Closeable put(MetricTag tag) {
        Set<MetricTag> set = getOrCreateLocalSet();
        set.add(tag);
        return () -> set.remove(tag);
    }

    @Nullable
    public static Set<MetricTag> getAll() {
        final Set<MetricTag> set = threadLocal.get();
        return set == null ? emptySet() : new HashSet<>(set);
    }

    private static Set<MetricTag> getOrCreateLocalSet() {
        Set<MetricTag> set = threadLocal.get();
        if (set == null) {
            threadLocal.set(set = new HashSet<>());
        }
        return set;
    }

    public interface Closeable extends AutoCloseable {
        @Override
        void close();
    }
}
