package com.atlassian.util.profiling;

import com.google.common.annotations.VisibleForTesting;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.concurrent.Immutable;
import java.util.Objects;

import static java.util.Objects.isNull;
import static java.util.Objects.requireNonNull;

/**
 * A data class representing a single metric tag, as a key-value pair.
 *
 * @since 3.4
 */
@Immutable
@ParametersAreNonnullByDefault
public final class MetricTag {
    public static final MetricTag SEND_ANALYTICS = new MetricTag("atl-analytics", "true");

    static final String FROM_PLUGIN_KEY_TAG_KEY = "fromPluginKey";
    static final String INVOKER_PLUGIN_KEY_TAG_KEY = "invokerPluginKey";
    @VisibleForTesting
    static final String UNDEFINED_TAG_VALUE = "undefined";

    private final String key;
    private final String value;

    private MetricTag(String key, @Nullable String value) {
        this.key = requireNonNull(key);
        this.value = isNull(value) ? UNDEFINED_TAG_VALUE : value;
    }

    @Nonnull
    public String getKey() {
        return key;
    }

    @Nonnull
    public String getValue() {
        return value;
    }

    @Override
    public String toString() {
        return key + "=" + value;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (!(o instanceof MetricTag)) return false;
        MetricTag metricTag = (MetricTag) o;
        return key.equals(metricTag.key) &&
                value.equals(metricTag.value);
    }

    @Override
    public int hashCode() {
        return Objects.hash(key, value);
    }

    /**
     * @apiNote nullable value since 3.6.1
     * @since 3.4.2
     */
    @Nonnull
    public static MetricTag of(String key, @Nullable String value) {
        return new MetricTag(key, value);
    }

    /**
     * @since 3.4.2
     */
    @Nonnull
    public static MetricTag of(String key, int value) {
        return new MetricTag(key, String.valueOf(value));
    }

    /**
     * @since 3.4.2
     */
    @Nonnull
    public static MetricTag of(String key, boolean value) {
        return new MetricTag(key, String.valueOf(value));
    }
}
