import {
    getListOfDeprecatedPlugins,
    __RewireAPI__ as Rewire,
} from 'health-checks/deprecated-plugins';
import { mockFetchFailure, mockFetchSuccess, restoreFetch } from '../fetch';

import deprecatedDescriptorsData from './data/deprecated-descriptors.data.json';
import deprecatedPluginData from './data/deprecated-plugin.data.json';
import deprecatedPluginData2 from './data/deprecated-plugin2.data.json';

const expectPluginData = (name = expect.any(String)) => {
    return expect.objectContaining({
        name,
        key: expect.any(String),
        links: expect.objectContaining({
            manage: expect.any(String),
        }),
    });
};

describe('getListOfDeprecatedPlugins', () => {
    afterEach(() => restoreFetch);

    it('Should return ref plugin and make one request for plugin data only given example data and on success', async () => {
        mockFetchSuccess(deprecatedDescriptorsData);
        mockFetchSuccess(deprecatedPluginData);

        const plugins = await getListOfDeprecatedPlugins();

        expect(plugins)
            .toBeArray()
            .toIncludeSameMembers([
                expectPluginData('Atlassian Jira - Plugins - Development Only - Reference Plugin'),
            ]);
    });

    it('Should return two plugins when DEPRECATED_PLUGINS_WHITELIST filters nothing', async () => {
        Rewire.__Rewire__('DEPRECATED_PLUGINS_WHITELIST', []);
        mockFetchSuccess(deprecatedDescriptorsData);
        mockFetchSuccess(deprecatedPluginData);
        mockFetchSuccess(deprecatedPluginData2);

        const plugins = await getListOfDeprecatedPlugins();

        expect(plugins)
            .toBeArray()
            .toIncludeSameMembers([
                expectPluginData('Atlassian Jira - Plugins - Development Only - Reference Plugin'),
                expectPluginData('JIRA Issue Collector Plugin'),
            ]);
    });

    it('Should return empty array of plugins and do not make extra requests when DEPRECATED_PLUGINS_WHITELIST filters everything', async () => {
        Rewire.__Rewire__('DEPRECATED_PLUGINS_WHITELIST', [
            'com.atlassian.jira.collector.plugin.jira-issue-collector-plugin',
            'com.atlassian.jira.dev.reference-plugin',
        ]);
        mockFetchSuccess(deprecatedDescriptorsData);

        const plugins = await getListOfDeprecatedPlugins();

        expect(plugins).toEqual([]);
    });

    /* Corrupted data */
    it('Should return empty array of plugins when `webresources` property is empty (1st endpoint data is corrupted)', async () => {
        mockFetchSuccess({});

        const plugins = await getListOfDeprecatedPlugins();

        expect(plugins).toEqual([]);
    });

    it('Should reject when there is no `moduleKey` property (1st endpoint data is corrupted)', done => {
        mockFetchSuccess({ webresources: [{}] });

        getListOfDeprecatedPlugins().catch(err => {
            expect(err).toBeInstanceOf(TypeError);
            done();
        });
    });

    it('Should reject when `moduleKey` property is not a string (1st endpoint data is corrupted)', done => {
        mockFetchSuccess({ webresources: [{ moduleKey: null }] });

        getListOfDeprecatedPlugins().catch(err => {
            expect(err).toBeInstanceOf(TypeError);
            done();
        });
    });

    it('Should return empty array of plugins and do not make extra requests when `moduleKey` property does not contain a colon (1st endpoint data is corrupted)', async () => {
        mockFetchSuccess({ webresources: [{ moduleKey: 'abc-dev' }, { moduleKey: 'def-plugin' }] });

        const plugins = await getListOfDeprecatedPlugins();

        expect(plugins).toEqual([]);
    });

    /* Failures */
    it('Should reject on fetchDeprecatedModules failure', done => {
        mockFetchSuccess({}, false);

        getListOfDeprecatedPlugins().catch(err => {
            expect(err).toBeInstanceOf(Error);
            done();
        });
    });

    it('Should reject when at least one of fetchPlugin requests fails', done => {
        mockFetchSuccess({ webresources: [{ moduleKey: 'abc:dev' }, { moduleKey: 'def:plugin' }] });
        mockFetchSuccess(deprecatedPluginData);
        mockFetchFailure(new TypeError('Failed to fetch'));

        getListOfDeprecatedPlugins().catch(err => {
            expect(err).toBeInstanceOf(TypeError);
            done();
        });
    });
});

describe('fetchPlugin', () => {
    const fetchPlugin = Rewire.__GetDependency__('fetchPlugin');

    afterEach(() => restoreFetch);

    it('Should call proper URL without cache', () => {
        const pluginKey = 'com.atlassian.jira.dev.reference-plugin';
        const fetchMock = mockFetchSuccess({});

        fetchPlugin(pluginKey);

        expect(fetchMock)
            .toHaveBeenCalledTimes(1)
            .toBeCalledWith(`/context/rest/plugins/1.0/${pluginKey}-key/summary`, {
                cache: 'no-cache',
            });
    });

    it('Should resolve with correct data on success', async () => {
        mockFetchSuccess(deprecatedPluginData);

        const modules = await fetchPlugin();

        expect(modules).toEqual(expectPluginData());
    });

    it('Should reject with error on server failure', done => {
        mockFetchSuccess(null, false);

        fetchPlugin().catch(err => {
            expect(err).toBeInstanceOf(Error);
            expect(err.message).toEqual('Fetch failed');
            done();
        });
    });

    it('Should reject on fetch failure', done => {
        mockFetchFailure(new TypeError('Failed to fetch'));

        fetchPlugin().catch(err => {
            expect(err).toBeInstanceOf(TypeError);
            expect(err.message).toEqual('Failed to fetch');
            done();
        });
    });
});

describe('fetchDeprecatedModules', () => {
    const fetchDeprecatedModules = Rewire.__GetDependency__('fetchDeprecatedModules');

    afterEach(() => restoreFetch);

    it('Should call proper URL without cache', () => {
        const fetchMock = mockFetchSuccess({});

        fetchDeprecatedModules();

        expect(fetchMock)
            .toHaveBeenCalledTimes(1)
            .toBeCalledWith('/context/rest/webResources/1.0/deprecatedDescriptors', {
                cache: 'no-cache',
            });
    });

    it('Should resolve with correct data on success', async () => {
        mockFetchSuccess(deprecatedDescriptorsData);

        const modules = await fetchDeprecatedModules();

        expect(modules).toEqual(
            expect.objectContaining({
                webresources: expect.any(Array),
            })
        );
        expect(modules.webresources).toSatisfyAll(wr => {
            return typeof wr.moduleKey === 'string' && wr.moduleKey.includes(':');
        });
    });

    it('Should reject with error on server failure', done => {
        mockFetchSuccess(null, false);

        fetchDeprecatedModules().catch(err => {
            expect(err).toBeInstanceOf(Error);
            expect(err.message).toEqual('Fetch failed');
            done();
        });
    });

    it('Should reject on fetch failure', done => {
        mockFetchFailure(new TypeError('Failed to fetch'));

        fetchDeprecatedModules().catch(err => {
            expect(err).toBeInstanceOf(TypeError);
            expect(err.message).toEqual('Failed to fetch');
            done();
        });
    });
});
