package com.atlassian.plugins.search.opensearch;

import com.atlassian.plugins.search.SearchResultsMarshaller;
import com.atlassian.plugins.search.SearchResultsMarshallerFactory;
import static com.atlassian.plugins.util.Check.notNull;
import com.atlassian.sal.api.ApplicationProperties;
import com.atlassian.sal.api.search.SearchProvider;
import com.atlassian.sal.api.search.SearchResults;
import com.atlassian.sal.api.user.UserManager;
import org.apache.commons.lang.StringUtils;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.text.MessageFormat;


/**
 * Servlet to provide search results as a feed.
 */
public class OpenSearchServlet extends HttpServlet
{
    private SearchProvider searchProvider;
    private UserManager userManager;
    private ApplicationProperties applicationProperties;
    private SearchResultsMarshallerFactory searchResultsMarshallerFactory;

    public static final String QUERY_PARAM = "query";
    public static final String START_PARAM = "start";
    public static final String COUNT_PARAM = "count";
    public static final String FEED_TYPE_PARAM = "format";
    public static final String IMPERSONATE_PARAM = "impersonate";


    public static final String DEFAULT_COUNT = "10";
    public static final String DEFAULT_FEED_TYPE = "rss_2.0";

    public static final String SEARCH_URL_ENCODING = "UTF-8";


    public OpenSearchServlet(SearchProvider searchProvider, UserManager userManager, ApplicationProperties applicationProperties, SearchResultsMarshallerFactory searchResultsMarshallerFactory)
    {
        this.searchProvider = notNull(searchProvider, "searchProvider");
        this.userManager = notNull(userManager, "userManager");
        this.applicationProperties = notNull(applicationProperties, "applicationProperties");
        this.searchResultsMarshallerFactory = notNull(searchResultsMarshallerFactory, "searchResultsMarshallerFactory");
    }

    @Override
    protected void doGet(HttpServletRequest req, HttpServletResponse resp) throws ServletException, IOException
    {
        String query = extractParam(req, QUERY_PARAM, null, true);
        String queryUrl = buildSearchUrl(query, req);
        SearchResults results = searchProvider.search(getSearchUser(req, resp), queryUrl);

        String feedType = extractParam(req, FEED_TYPE_PARAM, DEFAULT_FEED_TYPE, true);
        SearchResultsMarshaller marshaller = searchResultsMarshallerFactory.create(feedType);
        resp.setContentType(getContentType(feedType));
        marshaller.marshalTo(results, applicationProperties.getBaseUrl(), query, resp.getWriter());
    }

    /**
     * Gets the user which the search should be performed under. By default this will be the authenticated user, or
     * an impersonate parameter if sent by an admin user.
     * 
     * @param req the servlet request
     * @param resp the servlet response
     * @return the user to perform the search for
     * @throws IllegalArgumentException if the user to impersonate does not exist
     * @throws SecurityException if a non-admin user attempts to impersonate someone
     */
    protected String getSearchUser(HttpServletRequest req, HttpServletResponse resp) throws IllegalArgumentException, SecurityException
    {
        String impersonate = extractParam(req, IMPERSONATE_PARAM, null, false);

        if (impersonate == null) //no impersonation, just use current user
        {
            return userManager.getRemoteUsername();
        }
        else //requested impersonation
        {
            if (userManager.isSystemAdmin(userManager.getRemoteUsername()))
            {
                if (userManager.resolve(impersonate) == null)
                {
                    throw new IllegalArgumentException("User " + impersonate + " does not exist");
                }
                else return impersonate;
            }
            else
            {
                resp.setStatus(403);
                throw new SecurityException("User " + userManager.getRemoteUsername() + " cannot impersonate " + impersonate);
            }
        }
    }

    /**
     * Creates a search URL to be passed to SAL
     *
     * @param query the search query
     * @param req   the serlvet request
     * @return a URL-encoded search string
     */
    protected String buildSearchUrl(String query, HttpServletRequest req)
    {
        String count = extractParam(req, COUNT_PARAM, DEFAULT_COUNT, true);
        String start = extractParam(req, START_PARAM, "0", true);

        try
        {
            return MessageFormat.format("{0}&startIndex={1}&count={2}",
                    URLEncoder.encode(query, SEARCH_URL_ENCODING),
                    URLEncoder.encode(start, SEARCH_URL_ENCODING),
                    URLEncoder.encode(count, SEARCH_URL_ENCODING));
        }
        catch (UnsupportedEncodingException uee)
        {
            //this should never happen
            throw new RuntimeException("Failed to encode query URL", uee);
        }
    }

    /**
     * Extracts a string parameter from a servlet request. Does not allow blank parameters.
     *
     * @param req          the servlet request
     * @param key          the key to use to find the parameter
     * @param defaultValue a default value to use if the parameter cannot be found or is blank
     * @return the parameter value
     */
    protected String extractParam(HttpServletRequest req, String key, String defaultValue, boolean required)
    {
        String supplied = req.getParameter(key);

        if (!StringUtils.isBlank(supplied))
            return supplied;
        else if (defaultValue != null)
            return defaultValue;
        else if(required)
            throw new IllegalArgumentException("Parameter " + key + " must be supplied");
        else
            return null;
    }

    /**
     * Gets a content-type based upon a feed type
     *
     * @param feedType the type of feed
     * @return a content type
     */
    private String getContentType(String feedType)
    {
        //figure out which content type to use. If we don't know, fall back to text/xml
        if (feedType.startsWith("atom"))
            return "application/atom+xml";
        else if (feedType.startsWith("rss"))
            return "application/rss+xml";
        else
            return "text/xml";
    }
}
