package com.atlassian.plugins.rest.v2.multipart.fileupload;

import java.util.ArrayList;
import java.util.Collection;
import javax.servlet.http.HttpServletRequest;

import org.apache.commons.fileupload.FileItem;
import org.apache.commons.fileupload.FileUploadBase;
import org.apache.commons.fileupload.FileUploadException;
import org.apache.commons.fileupload.RequestContext;
import org.apache.commons.fileupload.disk.DiskFileItemFactory;
import org.apache.commons.fileupload.servlet.ServletFileUpload;
import org.apache.commons.fileupload.servlet.ServletRequestContext;

import com.atlassian.plugins.rest.api.multipart.FilePart;
import com.atlassian.plugins.rest.api.multipart.MultipartForm;
import com.atlassian.plugins.rest.api.multipart.MultipartHandler;
import com.atlassian.plugins.rest.v2.multipart.exception.FileCountLimitExceededException;
import com.atlassian.plugins.rest.v2.multipart.exception.FileSizeLimitExceededException;

public class CommonsFileUploadMultipartHandler implements MultipartHandler {

    public static final long NO_LIMIT = -1;
    public static final long DEFAULT_REQUEST_PART_LIMIT = 1000;
    private final ServletFileUpload servletFileUpload;

    public CommonsFileUploadMultipartHandler() {
        this(NO_LIMIT, NO_LIMIT, DEFAULT_REQUEST_PART_LIMIT);
    }

    /**
     * @param maxFileSize  Max file size, where -1 indicates no limit
     * @param maxSize      Max request size, where -1 indicates no limit
     * @param maxFileCount Max file count, where -1 indicates no limit
     */
    public CommonsFileUploadMultipartHandler(long maxFileSize, long maxSize, long maxFileCount) {
        servletFileUpload = new ServletFileUpload(new DiskFileItemFactory());
        servletFileUpload.setFileSizeMax(maxFileSize);
        servletFileUpload.setSizeMax(maxSize);
        servletFileUpload.setFileCountMax(maxFileCount);
    }

    @Override
    public FilePart getFilePart(HttpServletRequest request, String field) {
        return getForm(request).getFilePart(field);
    }

    private static class CommonsFileUploadMultipartForm implements MultipartForm {
        private final Collection<FileItem> fileItems;

        private CommonsFileUploadMultipartForm(final Collection<FileItem> fileItems) {
            this.fileItems = fileItems;
        }

        @Override
        public FilePart getFilePart(String field) {
            for (FileItem item : fileItems) {
                if (item.getFieldName().equals(field)) {
                    return new CommonsFileUploadFilePart(item);
                }
            }
            return null;
        }

        @Override
        public Collection<FilePart> getFileParts(String field) {
            Collection<FilePart> fileParts = new ArrayList<>();
            for (FileItem item : fileItems) {
                if (item.getFieldName().equals(field)) {
                    fileParts.add(new CommonsFileUploadFilePart(item));
                }
            }
            return fileParts;
        }
    }

    @Override
    public MultipartForm getForm(HttpServletRequest request) {
        return getForm(new ServletRequestContext(request));
    }

    public MultipartForm getForm(RequestContext request) {
        try {
            return new CommonsFileUploadMultipartForm(servletFileUpload.parseRequest(request));
        } catch (FileUploadException e) {
            if (e instanceof FileUploadBase.FileSizeLimitExceededException
                    || e instanceof FileUploadBase.SizeLimitExceededException) {
                throw new FileSizeLimitExceededException(e.getMessage());
            }
            if (e instanceof org.apache.commons.fileupload.FileCountLimitExceededException) {
                throw new FileCountLimitExceededException(e.getMessage());
            }
            throw new RuntimeException(e);
        }
    }
}
