package com.atlassian.plugins.rest.common.security.jersey;

import com.sun.jersey.api.model.AbstractMethod;
import com.sun.jersey.spi.container.ContainerRequest;
import com.sun.jersey.spi.container.ContainerRequestFilter;
import com.sun.jersey.spi.container.ContainerResponseFilter;
import com.sun.jersey.spi.container.ResourceFilter;

import com.atlassian.annotations.VisibleForTesting;
import com.atlassian.plugins.rest.common.security.AuthenticationRequiredException;
import com.atlassian.sal.api.features.DarkFeatureManager;
import com.atlassian.sal.api.user.UserKey;
import com.atlassian.sal.api.user.UserManager;

import static java.util.Objects.requireNonNull;

/**
 * This is a Jersey resource filter that checks whether the current client has access to current resource or its method. If the client doesn't
 * have access then an {@link AuthenticationRequiredException} is thrown.
 *
 * <p>
 * Resources can be marked as not needing authentication by using the {@link com.atlassian.plugins.rest.common.security.AnonymousSiteAccess} annotation
 */
class AuthenticatedResourceFilter implements ResourceFilter, ContainerRequestFilter {
    @VisibleForTesting
    static final String DEFAULT_TO_LICENSED_ACCESS_FEATURE_KEY = "atlassian.rest.default.to.licensed.access.disabled";

    static final String DEFAULT_TO_SYSADMIN_ACCESS_FEATURE_KEY = "atlassian.rest.default.to.sysadmin.access.enabled";
    static final String IGNORE_ANONYMOUSALLOWED_ANNOTATION =
            "atlassian.rest.ignore.anonymousallowed.annotation.enabled";
    private final AbstractMethod abstractMethod;
    private final UserManager userManager;
    private final DarkFeatureManager darkFeatureManager;

    public AuthenticatedResourceFilter(
            final AbstractMethod abstractMethod, UserManager userManager, final DarkFeatureManager darkFeatureManager) {
        this.abstractMethod = requireNonNull(abstractMethod, "abstractMethod can't be null");
        this.userManager = requireNonNull(userManager, "userManager can't be null");
        this.darkFeatureManager = requireNonNull(darkFeatureManager, "featureFlagManager can't be null");
    }

    public ContainerRequestFilter getRequestFilter() {
        return this;
    }

    public ContainerResponseFilter getResponseFilter() {
        return null;
    }

    public ContainerRequest filter(ContainerRequest request) {
        AccessType accessType = AccessType.getAccessType(abstractMethod);
        UserKey userKey = userManager.getRemoteUserKey();

        if (AccessType.EMPTY == accessType) {
            // DragonFly feature flag - unannotated resources should default to system admin access
            if (darkFeatureManager
                    .isEnabledForAllUsers(DEFAULT_TO_SYSADMIN_ACCESS_FEATURE_KEY)
                    .orElse(false)) {
                if (userManager.isSystemAdmin(userKey)) {
                    return request;
                }
            } else if (darkFeatureManager
                    .isEnabledForAllUsers(DEFAULT_TO_LICENSED_ACCESS_FEATURE_KEY)
                    .orElse(false)) {
                return request;
            } else if (userManager.isLicensed(userKey)) {
                return request;
            }
        } else if (AccessType.SYSTEM_ADMIN_ONLY == accessType) {
            if (userManager.isSystemAdmin(userKey)) {
                return request;
            }
        } else if (AccessType.ADMIN_ONLY == accessType) {
            if (userManager.isAdmin(userKey) || userManager.isSystemAdmin(userKey)) {
                return request;
            }
        } else if (AccessType.LICENSED_ONLY == accessType) {
            if (userManager.isLicensed(userKey)) {
                return request;
            }
        } else if (AccessType.UNLICENSED_SITE_ACCESS == accessType) {
            if (userManager.isLicensed(userKey) || userManager.isLimitedUnlicensedUser(userKey)) {
                return request;
            }
        } else if (AccessType.ANONYMOUS_SITE_ACCESS == accessType) {
            if ((userKey == null && userManager.isAnonymousAccessEnabled())
                    || userManager.isLicensed(userKey)
                    || userManager.isLimitedUnlicensedUser(userKey)) {
                return request;
            }
        } else if (AccessType.UNRESTRICTED_ACCESS == accessType) {
            return request;
            // Feature flag to ignore deprecated annotation
        } else if (!darkFeatureManager
                        .isEnabledForAllUsers(IGNORE_ANONYMOUSALLOWED_ANNOTATION)
                        .orElse(false)
                && AccessType.ANONYMOUS_ALLOWED == accessType) {
            return request;
        }

        throw new AuthenticationRequiredException();
    }
}
