package com.atlassian.plugins.rest.common.security.jersey;

import com.atlassian.plugins.rest.common.security.RequiresXsrfCheck;
import com.atlassian.sal.api.web.context.HttpContext;
import com.atlassian.sal.api.xsrf.XsrfTokenValidator;
import com.google.common.base.Preconditions;
import com.sun.jersey.api.model.AbstractMethod;
import com.sun.jersey.spi.container.ResourceFilter;
import com.sun.jersey.spi.container.ResourceFilterFactory;

import javax.ws.rs.ext.Provider;
import java.lang.annotation.Annotation;
import java.lang.reflect.AnnotatedElement;
import java.util.Collections;
import java.util.List;

/**
 * Factory for the XSRF resource filter
 *
 * @since 2.4
 */
@Provider
public class XsrfResourceFilterFactory implements ResourceFilterFactory
{
    private HttpContext httpContext;
    private XsrfTokenValidator xsrfTokenValidator;
    private static final String XSRF_ANNOTATION_CLASS_NAME =
        RequiresXsrfCheck.class.getSimpleName();

    public XsrfResourceFilterFactory(HttpContext httpContext, XsrfTokenValidator xsrfTokenValidator)
    {
        this.httpContext = Preconditions.checkNotNull(httpContext);
        this.xsrfTokenValidator = Preconditions.checkNotNull(xsrfTokenValidator);
    }

    public boolean hasXsrfAnnotation(AnnotatedElement annotatedElement)
    {
        if (annotatedElement.isAnnotationPresent(RequiresXsrfCheck.class))
        {
            return true;
        }
        for (Annotation annotation : annotatedElement.getAnnotations())
        {
            if (annotation.annotationType().getSimpleName().equals(
                XSRF_ANNOTATION_CLASS_NAME))
            {
                return true;
            }
        }
        return false;
    }

    public List<ResourceFilter> create(final AbstractMethod method)
    {
        if (hasXsrfAnnotation(method) || hasXsrfAnnotation(method.getResource()))
        {
            XsrfResourceFilter xsrfResourceFilter = new XsrfResourceFilter();
            xsrfResourceFilter.setHttpContext(httpContext);
            xsrfResourceFilter.setXsrfTokenValidator(xsrfTokenValidator);
            return Collections.<ResourceFilter>singletonList(xsrfResourceFilter);
        }
        return Collections.emptyList();
    }
}
