package com.atlassian.plugins.rest.common.security.jersey;

import com.atlassian.plugins.rest.common.security.XsrfCheckFailedException;
import com.atlassian.sal.api.web.context.HttpContext;
import com.atlassian.sal.api.xsrf.XsrfTokenAccessor;
import com.atlassian.sal.api.xsrf.XsrfTokenValidator;
import com.atlassian.sal.core.xsrf.IndependentXsrfTokenValidator;
import com.google.common.collect.ImmutableSet;
import com.sun.jersey.spi.container.ContainerRequest;
import com.sun.jersey.spi.container.ContainerRequestFilter;
import com.sun.jersey.spi.container.ContainerResponseFilter;
import com.sun.jersey.spi.container.ResourceFilter;

import javax.servlet.http.HttpServletRequest;
import javax.ws.rs.core.MediaType;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Locale;
import java.util.Set;


/**
 * A filter that filters requests that need XSRF protection.
 * <p/>
 * This checks for the presence of the no-check xsrf header and if the xsrf token is correct.
 *
 * @since 2.4
 */
public class XsrfResourceFilter implements ResourceFilter, ContainerRequestFilter
{
    public static final String TOKEN_HEADER = "X-Atlassian-Token";
    public static final String NO_CHECK = "no-check";


    private HttpContext httpContext;
    private XsrfTokenValidator xsrfTokenValidator;
    private static final Set<String> XSRFABLE_TYPES = new HashSet<String>(Arrays.asList(
            MediaType.APPLICATION_FORM_URLENCODED, MediaType.MULTIPART_FORM_DATA, MediaType.TEXT_PLAIN
    ));


    public void setHttpContext(HttpContext httpContext)
    {
        this.httpContext = httpContext;
    }

    public void setXsrfTokenValidator(final XsrfTokenValidator xsrfTokenValidator)
    {
        this.xsrfTokenValidator = xsrfTokenValidator;
    }

    public ContainerRequest filter(final ContainerRequest request)
    {
        if (isXsrfable(request))
        {
            String header = request.getHeaderValue(TOKEN_HEADER);
            if (header == null || !ImmutableSet.of(NO_CHECK, "nocheck").contains(header.toLowerCase(Locale.ENGLISH)))
            {
                HttpServletRequest httpServletRequest = null;
                if (httpContext != null)
                {
                    httpServletRequest = this.httpContext.getRequest();
                }
                if (!isXsrfTokenValid(httpServletRequest))
                {
                    throw new XsrfCheckFailedException();
                }
            }
        }
        return request;
    }

    private boolean isXsrfable(ContainerRequest request)
    {
        String method = request.getMethod();
        return method.equals("GET") || (method.equals("POST") && XSRFABLE_TYPES.contains(mediaTypeToString(request.getMediaType())));
    }

    public ContainerRequestFilter getRequestFilter()
    {
        return this;
    }

    public ContainerResponseFilter getResponseFilter()
    {
        return null;
    }

    private static String mediaTypeToString(MediaType mediaType)
    {
        return mediaType.getType().toLowerCase(Locale.ENGLISH) + "/" + mediaType.getSubtype().toLowerCase(Locale.ENGLISH);
    }

    /**
     * Returns true if the given request xsrf token cookie value
     * matches the xsrf token submitted in the request form.
     * Currently this method only works on requests that have a
     * media type of {@link MediaType.APPLICATION_FORM_URLENCODED_TYPE}.
     *
     * @param request the request to check.
     * @return true if the given request xsrf token cookie value
     * matches the xsrf token submitted in the request form.
     */
    protected boolean isXsrfTokenValid(HttpServletRequest httpServletRequest)
    {
        if (httpServletRequest == null)
        {
            return false;
        }
        return xsrfTokenValidator.validateFormEncodedToken(httpServletRequest);
    }

}
